% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_BKP.R, R/plot_DKP.R
\name{plot}
\alias{plot}
\alias{plot.BKP}
\alias{plot.DKP}
\title{Plot Fitted BKP or DKP Models}
\usage{
\method{plot}{BKP}(x, only_mean = FALSE, ...)

\method{plot}{DKP}(x, only_mean = FALSE, ...)
}
\arguments{
\item{x}{An object of class \code{"BKP"} or \code{"DKP"}, typically returned
by \code{\link{fit.BKP}} or \code{\link{fit.DKP}}.}

\item{only_mean}{Logical. If \code{TRUE}, only the predicted mean surface is
plotted for 2D inputs (only applies to \code{BKP} models). Default is
\code{FALSE}.}

\item{...}{Additional arguments passed to internal plotting routines
(currently unused).}
}
\value{
This function does not return a value. It is called for its side
effects, producing plots that visualize the model predictions and
uncertainty.
}
\description{
Visualizes fitted \code{BKP} or \code{DKP} models depending on
the input dimensionality. For 1-dimensional inputs, it displays predicted
class probabilities with credible intervals and observed data. For
2-dimensional inputs, it generates contour plots of posterior summaries.
}
\details{
The plotting behavior depends on the dimensionality of the input
covariates:

\itemize{
\item \strong{1D inputs:}
\itemize{
\item For \code{BKP}, the function plots the posterior mean curve with a 95\% credible band, along with the observed proportions (\eqn{y/m}).
\item For \code{DKP}, the function plots one curve per class, each with a shaded credible interval and observed multinomial class frequencies.
}

\item \strong{2D inputs:}
\itemize{
\item For both models, the function produces a 2-by-2 panel of contour plots for each class (or the success class in BKP), showing:
\enumerate{
\item Predictive mean surface
\item Predictive 97.5th percentile surface (upper bound of 95\% credible interval)
\item Predictive variance surface
\item Predictive 2.5th percentile surface (lower bound of 95\% credible interval)
}
}
}

For input dimensions greater than 2, the function will terminate with an
error.
}
\examples{
# ============================================================== #
# ========================= BKP Examples ======================= #
# ============================================================== #

#-------------------------- 1D Example ---------------------------
set.seed(123)

# Define true success probability function
true_pi_fun <- function(x) {
  (1 + exp(-x^2) * cos(10 * (1 - exp(-x)) / (1 + exp(-x)))) / 2
}

n <- 30
Xbounds <- matrix(c(-2,2), nrow=1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model1 <- fit.BKP(X, y, m, Xbounds=Xbounds)

# Plot results
plot(model1)


#-------------------------- 2D Example ---------------------------
set.seed(123)

# Define 2D latent function and probability transformation
true_pi_fun <- function(X) {
  if(is.null(nrow(X))) X <- matrix(X, nrow=1)
  m <- 8.6928
  s <- 2.4269
  x1 <- 4*X[,1]- 2
  x2 <- 4*X[,2]- 2
  a <- 1 + (x1 + x2 + 1)^2 *
    (19- 14*x1 + 3*x1^2- 14*x2 + 6*x1*x2 + 3*x2^2)
  b <- 30 + (2*x1- 3*x2)^2 *
    (18- 32*x1 + 12*x1^2 + 48*x2- 36*x1*x2 + 27*x2^2)
  f <- log(a*b)
  f <- (f- m)/s
  return(pnorm(f))  # Transform to probability
}

n <- 100
Xbounds <- matrix(c(0, 0, 1, 1), nrow = 2)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model2 <- fit.BKP(X, y, m, Xbounds=Xbounds)

# Plot results
plot(model2)

# ============================================================== #
# ========================= DKP Examples ======================= #
# ============================================================== #

#-------------------------- 1D Example ---------------------------
set.seed(123)

# Define true class probability function (3-class)
true_pi_fun <- function(X) {
  p <- (1 + exp(-X^2) * cos(10 * (1 - exp(-X)) / (1 + exp(-X)))) / 2
  return(matrix(c(p/2, p/2, 1 - p), nrow = length(p)))
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model1 <- fit.DKP(X, Y, Xbounds = Xbounds)

# Plot results
plot(model1)


#-------------------------- 2D Example ---------------------------
set.seed(123)

# Define latent function and transform to 3-class probabilities
true_pi_fun <- function(X) {
  if (is.null(nrow(X))) X <- matrix(X, nrow = 1)
  m <- 8.6928; s <- 2.4269
  x1 <- 4 * X[,1] - 2
  x2 <- 4 * X[,2] - 2
  a <- 1 + (x1 + x2 + 1)^2 *
    (19 - 14*x1 + 3*x1^2 - 14*x2 + 6*x1*x2 + 3*x2^2)
  b <- 30 + (2*x1 - 3*x2)^2 *
    (18 - 32*x1 + 12*x1^2 + 48*x2 - 36*x1*x2 + 27*x2^2)
  f <- (log(a * b) - m) / s
  p <- pnorm(f)
  return(matrix(c(p/2, p/2, 1 - p), nrow = length(p)))
}

n <- 100
Xbounds <- matrix(c(0, 0, 1, 1), nrow = 2)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model2 <- fit.DKP(X, Y, Xbounds = Xbounds)

# Plot results
plot(model2)

}
\seealso{
\code{\link{fit.BKP}}, \code{\link{predict.BKP}},
\code{\link{fit.DKP}}, \code{\link{predict.DKP}}
}
\keyword{BKP}
\keyword{DKP}
