% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blob_client_funcs.R
\name{list_blobs}
\alias{list_blobs}
\alias{upload_blob}
\alias{multiupload_blob}
\alias{download_blob}
\alias{multidownload_blob}
\alias{delete_blob}
\title{Operations on a blob container or blob}
\usage{
list_blobs(container, info = c("partial", "name", "all"),
  prefix = NULL)

upload_blob(container, src, dest, type = "BlockBlob", blocksize = 2^24,
  lease = NULL, use_azcopy = FALSE)

multiupload_blob(container, src, dest, type = "BlockBlob",
  blocksize = 2^24, lease = NULL, use_azcopy = FALSE,
  max_concurrent_transfers = 10)

download_blob(container, src, dest, overwrite = FALSE, lease = NULL,
  use_azcopy = FALSE)

multidownload_blob(container, src, dest, overwrite = FALSE,
  lease = NULL, use_azcopy = FALSE, max_concurrent_transfers = 10)

delete_blob(container, blob, confirm = TRUE)
}
\arguments{
\item{container}{A blob container object.}

\item{info}{For \code{list_blobs}, level of detail about each blob to return: a vector of names only; the name, size and last-modified date (default); or all information.}

\item{prefix}{For \code{list_blobs}, filters the result to return only blobs whose name begins with this prefix.}

\item{src, dest}{The source and destination files for uploading and downloading. See 'Details' below.For uploading, \code{src} can also be a \link{textConnection} or \link{rawConnection} object to allow transferring in-memory R objects without creating a temporary file. For downloading,}

\item{type}{When uploading, the type of blob to create. Currently only block blobs are supported.}

\item{blocksize}{The number of bytes to upload per HTTP(S) request.}

\item{lease}{The lease for a blob, if present.}

\item{use_azcopy}{Whether to use the AzCopy utility from Microsoft to do the transfer, rather than doing it in R.}

\item{max_concurrent_transfers}{For \code{multiupload_blob} and \code{multidownload_blob}, the maximum number of concurrent file transfers. Each concurrent file transfer requires a separate R process, so limit this if you are low on memory.}

\item{overwrite}{When downloading, whether to overwrite an existing destination file.}

\item{blob}{A string naming a blob.}

\item{confirm}{Whether to ask for confirmation on deleting a blob.}
}
\value{
For \code{list_blobs}, details on the blobs in the container. For \code{download_blob}, if \code{dest=NULL}, the contents of the downloaded blob as a raw vector.
}
\description{
Upload, download, or delete a blob; list blobs in a container.
}
\details{
\code{upload_blob} and \code{download_blob} are the workhorse file transfer functions for blobs. They each take as inputs a \emph{single} filename or connection as the source for uploading/downloading, and a single filename as the destination.

\code{multiupload_blob} and \code{multidownload_blob} are functions for uploading and downloading \emph{multiple} blobs at once. They parallelise file transfers by deploying a pool of R processes in the background, which can lead to significantly greater efficiency when transferring many small files. They take as input a wildcard pattern as the source, which expands to one or more files. The \code{dest} argument should be a directory.

The file transfer functions also support working with connections to allow transferring R objects without creating temporary files. For uploading, \code{src} can be a \link{textConnection} or \link{rawConnection} object. For downloading, \code{dest} can be NULL or a \code{rawConnection} object. In the former case, the downloaded data is returned as a raw vector, and for the latter, it will be placed into the connection. See the examples below.

By default, \code{download_blob} will display a progress bar as it is downloading. To turn this off, use \code{options(azure_dl_progress_bar=FALSE)}. To turn the progress bar back on, use \code{options(azure_dl_progress_bar=TRUE)}.
}
\examples{
\dontrun{

cont <- blob_container("https://mystorage.blob.core.windows.net/mycontainer", key="access_key")

list_blobs(cont)

upload_blob(cont, "~/bigfile.zip", dest="bigfile.zip")
download_blob(cont, "bigfile.zip", dest="~/bigfile_downloaded.zip")

delete_blob(cont, "bigfile.zip")

# uploading/downloading multiple files at once
multiupload_blob(cont, "/data/logfiles/*.zip", "/uploaded_data")
multiupload_blob(cont, "myproj/*")  # no dest directory uploads to root
multidownload_blob(cont, "jan*.*", "/data/january")

# uploading serialized R objects via connections
json <- jsonlite::toJSON(iris, pretty=TRUE, auto_unbox=TRUE)
con <- textConnection(json)
upload_blob(cont, con, "iris.json")

rds <- serialize(iris, NULL)
con <- rawConnection(rds)
upload_blob(cont, con, "iris.rds")

# downloading files into memory: as a raw vector, and via a connection
rawvec <- download_blob(cont, "iris.json", NULL)
rawToChar(rawvec)

con <- rawConnection(raw(0), "r+")
download_blob(cont, "iris.rds", con)
unserialize(con)

}
}
\seealso{
\link{blob_container}, \link{az_storage}, \link{storage_download}, \link{call_azcopy}

\href{https://github.com/Azure/azure-storage-azcopy}{AzCopy version 10 on GitHub}
}
