% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/managed_token.R, R/token.R
\name{get_managed_token}
\alias{get_managed_token}
\alias{get_azure_token}
\alias{delete_azure_token}
\alias{load_azure_token}
\alias{clean_token_directory}
\alias{list_azure_tokens}
\alias{token_hash}
\alias{is_azure_token}
\alias{is_azure_v1_token}
\alias{is_azure_v2_token}
\title{Manage Azure Active Directory OAuth 2.0 tokens}
\usage{
get_managed_token(resource, token_args = list(), use_cache = TRUE)

get_azure_token(
  resource,
  tenant,
  app,
  password = NULL,
  username = NULL,
  certificate = NULL,
  auth_type = NULL,
  aad_host = "https://login.microsoftonline.com/",
  version = 1,
  authorize_args = list(),
  token_args = list(),
  use_cache = TRUE,
  on_behalf_of = NULL,
  auth_code = NULL,
  device_creds = NULL
)

delete_azure_token(
  resource,
  tenant,
  app,
  password = NULL,
  username = NULL,
  certificate = NULL,
  auth_type = NULL,
  aad_host = "https://login.microsoftonline.com/",
  version = 1,
  authorize_args = list(),
  token_args = list(),
  on_behalf_of = NULL,
  hash = NULL,
  confirm = TRUE
)

load_azure_token(hash)

clean_token_directory(confirm = TRUE)

list_azure_tokens()

token_hash(
  resource,
  tenant,
  app,
  password = NULL,
  username = NULL,
  certificate = NULL,
  auth_type = NULL,
  aad_host = "https://login.microsoftonline.com/",
  version = 1,
  authorize_args = list(),
  token_args = list(),
  on_behalf_of = NULL
)

is_azure_token(object)

is_azure_v1_token(object)

is_azure_v2_token(object)
}
\arguments{
\item{resource}{For AAD v1.0, the URL of your resource host, or a GUID. For AAD v2.0, a character vector of scopes, each consisting of a URL or GUID along with a path designating the access scope. See 'Details' below.}

\item{token_args}{An optional list of further parameters for the token endpoint. These will be included in the body of the request for \code{get_azure_token}, or as URI query parameters for \code{get_managed_token}.}

\item{use_cache}{If TRUE and cached credentials exist, use them instead of obtaining a new token. Set this to FALSE to bypass the cache.}

\item{tenant}{Your tenant. This can be a name ("myaadtenant"), a fully qualified domain name ("myaadtenant.onmicrosoft.com" or "mycompanyname.com"), or a GUID. It can also be one of the generic tenants "common", "organizations" or "consumers"; see 'Generic tenants' below.}

\item{app}{The client/app ID to use to authenticate with.}

\item{password}{For most authentication flows, this is the password for the \emph{app} where needed, also known as the client secret. For the resource owner grant, this is your personal account password. See 'Details' below.}

\item{username}{Your AAD username, if using the resource owner grant. See 'Details' below.}

\item{certificate}{A file containing the certificate for authenticating with (including the private key), an Azure Key Vault certificate object, or a call to the \code{cert_assertion} function to build a client assertion with a certificate. See 'Certificate authentication' below.}

\item{auth_type}{The authentication type. See 'Details' below.}

\item{aad_host}{URL for your AAD host. For the public Azure cloud, this is \verb{https://login.microsoftonline.com/}. Change this if you are using a government or private cloud. Can also be a full URL, eg \verb{https://mydomain.b2clogin.com/mydomain/other/path/names/oauth2} (this is relevant mainly for Azure B2C logins).}

\item{version}{The AAD version, either 1 or 2. Authenticating with a personal account as opposed to a work or school account requires AAD 2.0.}

\item{authorize_args}{An optional list of further parameters for the AAD authorization endpoint. These will be included in the request URI as query parameters. Only used if \code{auth_type="authorization_code"}.}

\item{on_behalf_of}{For the on-behalf-of authentication type, a token. This should be either an AzureToken object, or a string containing the JWT-encoded token itself.}

\item{auth_code}{For the \code{authorization_code} flow, the code. Only used if \code{auth_type == "authorization_code"}.}

\item{device_creds}{For the \code{device_code} flow, the device credentials used to verify the session between the client and the server. Only used if \code{auth_type == "device_code"}.}

\item{hash}{The MD5 hash of this token, computed from the above inputs. Used by \code{load_azure_token} and \code{delete_azure_token} to identify a cached token to load and delete, respectively.}

\item{confirm}{For \code{delete_azure_token}, whether to prompt for confirmation before deleting a token.}

\item{object}{For \code{is_azure_token}, \code{is_azure_v1_token} and \code{is_azure_v2_token}, an R object.}
}
\description{
Use these functions to authenticate with Azure Active Directory (AAD).
}
\details{
\code{get_azure_token} does much the same thing as \code{\link[httr:oauth2.0_token]{httr::oauth2.0_token()}}, but customised for Azure. It obtains an OAuth token, first by checking if a cached value exists on disk, and if not, acquiring it from the AAD server. \code{load_azure_token} loads a token given its hash, \code{delete_azure_token} deletes a cached token given either the credentials or the hash, and \code{list_azure_tokens} lists currently cached tokens.

\code{get_managed_token} is a specialised function to acquire tokens for a \emph{managed identity}. This is an Azure service, such as a VM or container, that has been assigned its own identity and can be granted access permissions like a regular user. The advantage of managed identities over the other authentication methods (see below) is that you don't have to store a secret password, which improves security. Note that \code{get_managed_token} can only be used from within the managed identity itself.

The \code{resource} arg should be a single URL or GUID for AAD v1.0. For AAD v2.0, it should be a vector of \emph{scopes}, where each scope consists of a URL or GUID along with a path that designates the type of access requested. If a v2.0 scope doesn't have a path, \code{get_azure_token} will append the \verb{/.default} path with a warning. A special scope is \code{offline_access}, which requests a refresh token from AAD along with the access token: without this scope, you will have to reauthenticate if you want to refresh the token.

The \code{auth_code} and \code{device_creds} arguments are intended for use in embedded scenarios, eg when AzureAuth is loaded from within a Shiny web app. They enable the flow authorization step to be separated from the token acquisition step, which is necessary within an app; you can generally ignore these arguments when using AzureAuth interactively or as part of an R script. See the help for \link{build_authorization_uri} for examples on their use.

\code{token_hash} computes the MD5 hash of its arguments. This is used by AzureAuth to identify tokens for caching purposes. Note that tokens are only cached if you allowed AzureAuth to create a data directory at package startup.

One particular use of the \code{authorize_args} argument is to specify a different redirect URI to the default; see the examples below.
}
\section{Authentication methods}{

\enumerate{
\item Using the \strong{authorization_code} method is a multi-step process. First, \code{get_azure_token} opens a login window in your browser, where you can enter your AAD credentials. In the background, it loads the \href{https://github.com/rstudio/httpuv}{httpuv} package to listen on a local port. Once you have logged in, the AAD server redirects your browser to a local URL that contains an authorization code. \code{get_azure_token} retrieves this authorization code and sends it to the AAD access endpoint, which returns the OAuth token.
\item The \strong{device_code} method is similar in concept to authorization_code, but is meant for situations where you are unable to browse the Internet -- for example if you don't have a browser installed or your computer has input constraints. First, \code{get_azure_token} contacts the AAD devicecode endpoint, which responds with a login URL and an access code. You then visit the URL and enter the code, possibly using a different computer. Meanwhile, \code{get_azure_token} polls the AAD access endpoint for a token, which is provided once you have entered the code.
\item The \strong{client_credentials} method is much simpler than the above methods, requiring only one step. \code{get_azure_token} contacts the access endpoint, passing it either the app secret or the certificate assertion (which you supply in the \code{password} or \code{certificate} argument respectively). Once the credentials are verified, the endpoint returns the token. This is the method typically used by service accounts.
\item The \strong{resource_owner} method also requires only one step. In this method, \code{get_azure_token} passes your (personal) username and password to the AAD access endpoint, which validates your credentials and returns the token.
\item The \strong{on_behalf_of} method is used to authenticate with an Azure resource by passing a token obtained beforehand. It is mostly used by intermediate apps to authenticate for users. In particular, you can use this method to obtain tokens for multiple resources, while only requiring the user to authenticate once: see the examples below.
}

If the authentication method is not specified, it is chosen based on the presence or absence of the other arguments, and whether httpuv is installed.

The httpuv package must be installed to use the authorization_code method, as this requires a web server to listen on the (local) redirect URI. See \link[httr:oauth2.0_token]{httr::oauth2.0_token} for more information; note that Azure does not support the \code{use_oob} feature of the httr OAuth 2.0 token class.

Similarly, since the authorization_code method opens a browser to load the AAD authorization page, your machine must have an Internet browser installed that can be run from inside R. In particular, if you are using a Linux \href{https://azure.microsoft.com/en-us/services/virtual-machines/data-science-virtual-machines/}{Data Science Virtual Machine} in Azure, you may run into difficulties; use one of the other methods instead.
}

\section{Certificate authentication}{

OAuth tokens can be authenticated via an SSL/TLS certificate, which is considered more secure than a client secret. To do this, use the \code{certificate} argument, which can contain any of the following:
\itemize{
\item The name of a PEM or PFX file, containing \emph{both} the private key and the public certificate.
\item A certificate object from the AzureKeyVault package, representing a cert stored in the Key Vault service.
\item A call to the \code{cert_assertion()} function to customise details of the requested token, eg the duration, expiry date, custom claims, etc. See the examples below.
}
}

\section{Generic tenants}{


There are 3 generic values that can be used as tenants when authenticating:\tabular{ll}{
   Tenant \tab Description \cr
   \code{common} \tab Allows users with both personal Microsoft accounts and work/school accounts from Azure AD to sign into the application. \cr
   \code{organizations} \tab Allows only users with work/school accounts from Azure AD to sign into the application. \cr
   \code{consumers} \tab Allows only users with personal Microsoft accounts (MSA) to sign into the application. \cr
}
}

\section{OpenID Connect}{

\code{get_azure_token} can be used to obtain ID tokens along with regular OAuth access tokens, when using an interactive authentication flow (authorization_code or device_code). The behaviour depends on the AAD version:
\itemize{
\item AAD v1.0 will return an ID token as well as the access token by default; you don't have to do anything extra. However, AAD v1.0 will not \emph{refresh} the ID token when it expires; you must reauthenticate to get a new one. To ensure you don't pull the cached version of the credentials, specify \code{use_cache=FALSE} in the calls to \code{get_azure_token}.
\item Unlike AAD v1.0, AAD v2.0 does not return an ID token by default. To get a token, specify \code{openid} as a scope. On the other hand it \emph{does} refresh the ID token, so bypassing the cache is not needed. It's recommended to use AAD v2.0 if you only want an ID token.
}
}

\section{Caching}{

AzureAuth caches tokens based on all the inputs to \code{get_azure_token} as listed above. Tokens are cached in a custom, user-specific directory, created with the rappdirs package. On recent Windows versions, this will usually be in the location \verb{C:\\\\Users\\\\(username)\\\\AppData\\\\Local\\\\AzureR}. On Linux, it will be in \verb{~/.config/AzureR}, and on MacOS, it will be in \verb{~/Library/Application Support/AzureR}. Note that a single directory is used for all tokens, and the working directory is not touched (which significantly lessens the risk of accidentally introducing cached tokens into source control).

To list all cached tokens on disk, use \code{list_azure_tokens}. This returns a list of token objects, named according to their MD5 hashes.

To delete a cached token, use \code{delete_azure_token}. This takes the same inputs as \code{get_azure_token}, or you can specify the MD5 hash directly in the \code{hash} argument.

To delete \emph{all} cached tokens, use \code{clean_token_directory}.
}

\section{Refreshing}{

A token object can be refreshed by calling its \code{refresh()} method. If the token's credentials contain a refresh token, this is used; otherwise a new access token is obtained by reauthenticating.

Note that in AAD, a refresh token can be used to obtain an access token for any resource or scope that you have permissions for. Thus, for example, you could use a refresh token issued on a request for \verb{https://management.azure.com} to obtain a new access token for \verb{https://graph.microsoft.com} (assuming you've been granted permission).

To obtain an access token for a new resource, change the object's \code{resource} (for an AAD v1.0 token) or \code{scope} field (for an AAD v2.0 token) before calling \code{refresh()}. If you \emph{also} want to retain the token for the old resource, you should call the \code{clone()} method first to create a copy. See the examples below.
}

\section{Value}{

For \code{get_azure_token}, an object inheriting from \code{AzureToken}. The specific class depends on the authentication flow: \code{AzureTokenAuthCode}, \code{AzureTokenDeviceCode}, \code{AzureTokenClientCreds}, \code{AzureTokenOnBehalfOf}, \code{AzureTokenResOwner}. For \code{get_managed_token}, a similar object of class \code{AzureTokenManaged}.

For \code{list_azure_tokens}, a list of such objects retrieved from disk.
}

\examples{
\dontrun{

# authenticate with Azure Resource Manager:
# no user credentials are supplied, so this will use the authorization_code
# method if httpuv is installed, and device_code if not
get_azure_token("https://management.azure.com/", tenant="mytenant", app="app_id")

# you can force a specific authentication method with the auth_type argument
get_azure_token("https://management.azure.com/", tenant="mytenant", app="app_id",
    auth_type="device_code")

# to default to the client_credentials method, supply the app secret as the password
get_azure_token("https://management.azure.com/", tenant="mytenant", app="app_id",
    password="app_secret")

# authenticate to your resource with the resource_owner method: provide your username and password
get_azure_token("https://myresource/", tenant="mytenant", app="app_id",
    username="user", password="abcdefg")

# obtaining multiple tokens: authenticate (interactively) once...
tok0 <- get_azure_token("serviceapp_id", tenant="mytenant", app="clientapp_id",
    auth_type="authorization_code")
# ...then get tokens for each resource (Resource Manager and MS Graph) with on_behalf_of
tok1 <- get_azure_token("https://management.azure.com/", tenant="mytenant", app="serviceapp_id",
    password="serviceapp_secret", on_behalf_of=tok0)
tok2 <- get_azure_token("https://graph.microsoft.com/", tenant="mytenant", app="serviceapp_id",
    password="serviceapp_secret", on_behalf_of=tok0)


# authorization_code flow with app registered in AAD as a web rather than a native client:
# supply the client secret in the password arg
get_azure_token("https://management.azure.com/", "mytenant", "app_id",
    password="app_secret", auth_type="authorization_code")


# use a different redirect URI to the default localhost:1410
get_azure_token("https://management.azure.com/", tenant="mytenant", app="app_id",
    authorize_args=list(redirect_uri="http://localhost:8000"))


# request an AAD v1.0 token for Resource Manager (the default)
token1 <- get_azure_token("https://management.azure.com/", "mytenant", "app_id")

# same request to AAD v2.0, along with a refresh token
token2 <- get_azure_token(c("https://management.azure.com/.default", "offline_access"),
    "mytenant", "app_id", version=2)

# requesting multiple scopes (Microsoft Graph) with AAD 2.0
get_azure_token(c("https://graph.microsoft.com/User.Read.All",
                  "https://graph.microsoft.com/User.ReadWrite.All",
                  "https://graph.microsoft.com/Directory.ReadWrite.All",
                  "offline_access"),
    "mytenant", "app_id", version=2)


# list saved tokens
list_azure_tokens()

# delete a saved token from disk
delete_azure_token(resource="https://myresource/", tenant="mytenant", app="app_id",
    username="user", password="abcdefg")

# delete a saved token by specifying its MD5 hash
delete_azure_token(hash="7ea491716e5b10a77a673106f3f53bfd")


# authenticating for B2C logins (custom AAD host)
get_azure_token("https://mydomain.com", "mytenant", "app_id", "password",
    aad_host="https://mytenant.b2clogin.com/tfp/mytenant.onmicrosoft.com/custom/oauth2")


# authenticating with a certificate
get_azure_token("https://management.azure.com/", "mytenant", "app_id",
    certificate="mycert.pem")

# authenticating with a certificate stored in Azure Key Vault
cert <- AzureKeyVault::key_vault("myvault")$certificates$get("mycert")
get_azure_token("https://management.azure.com/", "mytenant", "app_id",
    certificate=cert)

# get a token valid for 2 hours (default is 1 hour)
get_azure_token("https://management.azure.com/", "mytenant", "app_id",
    certificate=cert_assertion("mycert.pem", duration=2*3600))


# ID token with AAD v1.0
# if you only want an ID token, set the resource to blank ("")
tok <- get_azure_token("", "mytenant", "app_id", use_cache=FALSE)
extract_jwt(tok, "id")

# ID token with AAD v2.0 (recommended)
tok2 <- get_azure_token(c("openid", "offline_access"), "mytenant", "app_id", version=2)
extract_jwt(tok2, "id")


# get a token from within a managed identity (VM, container or service)
get_managed_token("https://management.azure.com/")


# use a refresh token from one resource to get an access token for another resource
tok <- get_azure_token("https://myresource", "mytenant", "app_id")
tok2 <- tok$clone()
tok2$resource <- "https://anotherresource"
tok2$refresh()

# same for AAD v2.0
tok <- get_azure_token(c("https://myresource/.default", "offline_access"),
    "mytenant", "app_id", version=2)
tok2 <- tok$clone()
tok2$scope <- c("https://anotherresource/.default", "offline_access")
tok2$refresh()

}
}
\seealso{
\link{AzureToken}, \link[httr:oauth2.0_token]{httr::oauth2.0_token}, \link[httr:Token-class]{httr::Token}, \link{cert_assertion},
\link{build_authorization_uri}, \link{get_device_creds}

\href{https://docs.microsoft.com/en-us/azure/active-directory/develop/}{Azure Active Directory for developers},
\href{https://www.oauth.com/oauth2-servers/device-flow/token-request/}{Device code flow on OAuth.com},
\href{https://tools.ietf.org/html/rfc6749}{OAuth 2.0 RFC} for the gory details on how OAuth works
}
