% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/AF20150824.R
\name{AF.ch}
\alias{AF.ch}
\title{Attributable fraction function for cohort sampling designs with time-to-event outcomes.}
\usage{
AF.ch(formula, data, exposure, ties = "breslow", times, clusterid)
}
\arguments{
\item{formula}{a formula object, with the response on the left of a ~ operator, and the terms on the right. The response must be a survival object as returned by the \code{Surv} function (\code{\link[survival]{Surv}}). The exposure and confounders should be specified as independent (right-hand side) variables. The time-to-event outcome should be specified by the survival object. The formula is used to fit a Cox proportional hazards model.}

\item{data}{an optional data frame, list or environment (or object coercible by \code{as.data.frame} to a data frame) containing the variables in the model. If not found in data, the variables are taken from environment (\code{formula}), typically the environment from which the function is called.}

\item{exposure}{the name of the exposure variable as a string. The exposure must be binary (0/1) where unexposed is coded as 0.}

\item{ties}{a character string specifying the method for tie handling. If there are no tied death times all the methods are equivalent. Uses the Breslow method by default.}

\item{times}{a scalar or vector of time points specified by the user for which the attributable fraction function is estimated. If not specified the observed death times will be used.}

\item{clusterid}{the name of the cluster identifier variable as a string, if data are clustered.}
}
\value{
\item{AF.est}{estimated attributable fraction function for every time point specified by \code{times}.}

\item{AF.var}{estimated variance of \code{AF.est}. The variance is obtained by combining the delta methods with the sandwich formula.}

\item{S.est}{estimated factual survival function; \eqn{S(t)}.}

\item{S.var}{estimated variance of \code{S.est}. The variance is obtained by the sandwich formula.}

\item{S0.est}{estimated counterfactual survival function if exposure would be eliminated; \eqn{S_0(t)}{S0(t)}.}

\item{S0.var}{estimated variance of \code{S0.est}. The variance is obtained by the sandwich formula.}

\item{fit}{the fitted model. Fitted using Cox proportional hazard, \code{\link[survival]{coxph}}.}
}
\description{
\code{AF.ch} estimates the model-based adjusted attributable fraction function for data from cohort sampling designs with time-to-event outcomes.
}
\details{
\code{Af.ch} estimates the attributable fraction for a time-to-event outcome
under the hypothetical scenario where a binary exposure \code{X} is eliminated from the population. The estimate is adjusted for confounders \code{Z}
by the Cox proportional hazards model (\code{\link[survival]{coxph}}). Let the AF function be defined as
\deqn{AF=1-\frac{\{1-S_0(t)\}}{\{1-S(t)\}}}{AF = 1 - {1 - S0(t)} / {1 - S(t)}}
where \eqn{S_0(t)}{S0(t)} denotes the counterfactual survival function for the event if
the exposure would have been eliminated from the population at baseline and \eqn{S(t)} denotes the factual survival function.
If \code{Z} is sufficient for confounding control, then \eqn{S_0(t)}{S0(t)} can be expressed as \eqn{E_Z\{S(t\mid{X=0,Z })\}}{E_z{S(t|X=0,Z)}}.
The function uses Cox proportional hazards regression to estimate \eqn{S(t\mid{X=0,Z})}{S(t|X=0,Z)}, and the marginal sample distribution of \code{Z}
to approximate the outer expectation (Sjolander and Vansteelandt, 2014).  If \code{clusterid} is supplied, then a clustered sandwich formula is used in all variance calculations.
}
\examples{
# Simulate a sample from a cohort sampling design with time-to-event outcome
expit <- function(x) 1 / (1 + exp( - x))
n <- 500
time <- c(seq(from = 0.2, to = 1, by = 0.2))
Z <- rnorm(n = n)
X <- rbinom(n = n, size = 1, prob = expit(Z))
Tim <- rexp(n = n, rate = exp(X + Z))
C <- rexp(n = n, rate = exp(X + Z))
Tobs <- pmin(Tim, C)
D <- as.numeric(Tobs < C)
#Ties created by rounding
Tobs <- round(Tobs, digits = 2)

# Example 1: non clustered data from a cohort sampling design with time-to-event outcomes
data <- data.frame(Tobs, D, X,  Z)
AF.est.ch <- AF.ch(formula = Surv(Tobs, D) ~ X + Z + X * Z, data = data,
                   exposure = "X", times = time)
summary(AF.est.ch)

# Example 2: clustered data from a cohort sampling design with time-to-event outcomes
# Duplicate observations in order to create clustered data
id <- rep(1:n, 2)
data <- data.frame(Tobs = c(Tobs, Tobs), D = c(D, D), X = c(X, X), Z = c(Z, Z), id = id)
AF.est.ch.clust <- AF.ch(formula = Surv(Tobs, D) ~ X + Z + X * Z, data = data,
                         exposure = "X", times = time, clusterid = "id")
summary(AF.est.ch.clust)
plot(AF.est.ch.clust, CI = TRUE)
}
\author{
Elisabeth Dahlqwist, Arvid Sjolander
}
\references{
Chen, L., Lin, D. Y., and Zeng, D. (2010). Attributable fraction functions for censored event times. \emph{Biometrika} \bold{97}, 713-726.

Sjolander, A. and Vansteelandt, S. (2014). Doubly robust estimation of attributable fractions in survival analysis. \emph{Statistical Methods in Medical Research}. doi: 10.1177/0962280214564003.
}
\seealso{
\code{\link[survival]{coxph}} and \code{\link[survival]{Surv}} used for fitting the Cox proportional hazards model.
}

