% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_run_modify.r
\name{umxThresholdMatrix}
\alias{umxThresholdMatrix}
\title{Create the threshold matrix needed for modeling ordinal data.}
\usage{
umxThresholdMatrix(df, sep = NA, method = c("auto", "Mehta", "allFree"),
  thresholds = c("deviationBased", "direct", "ignore", "left_censored"),
  threshMatName = "threshMat", l_u_bound = c(NA, NA), droplevels = FALSE,
  suffixes = "deprecated", verbose = FALSE)
}
\arguments{
\item{df}{the data being modeled (to allow access to the factor levels and quantiles within these for each variable)}

\item{sep}{(optional) string for wide (twin) data, separating the base name from a numeric suffix (e.g. "_T")}

\item{method}{How to set the thresholds: auto (the default), Mehta, which fixes the first two (auto chooses this for ordinal) or "allFree" (auto chooses this for binary)}

\item{thresholds}{How to implement thresholds: "deviationBased" (default), "direct", "ignore", "left_censored"}

\item{threshMatName}{name of the matrix which is returned. Defaults to "threshMat" - best not to change it.}

\item{l_u_bound}{c(NA, NA) by default, you can use this to bound the thresholds. Careful you don't set bounds too close if you do.}

\item{droplevels}{Whether to drop levels with no observed data (defaults to FALSE)}

\item{suffixes}{deprecated. Instead of c("T1", "T2"), set sep (see above)}

\item{verbose}{(defaults to FALSE))}
}
\value{
- thresholds matrix
}
\description{
High-level helper for ordinal modeling. Creates, labels, and sets smart-starts for this complex matrix. Big time saver!
}
\details{
When modeling ordinal data (sex, low-med-hi, 
depressed/normal, not at all, rarely, often, always), a useful conceptual strategy to handle expectations
is to build a standard-normal model (i.e., a latent model with zero-means, and unit (1.0) variances),
and then to threshold this normal distribution to generate the observed data. Thus an observation of "depressed"
is modeled as a high score on the latent normally distributed trait, with thresholds set so that only scores above
this threshold (1-minus the number of categories).

For \strong{deviation methods}, it returns a list of lowerOnes_for_thresh, deviations_for_thresh & thresholdsAlgebra (named threshMatName)

For \strong{direct}, it returns a thresholdsMatrix (named threshMatName)
}
\examples{
x = data.frame(ordered(rbinom(100,1,.5))); names(x) <- c("x")
umxThresholdMatrix(x)
x = cut(rnorm(100), breaks = c(-Inf,.2,.5, .7, Inf)); levels(x) = 1:5
x = data.frame(ordered(x)); names(x) <- c("x")
tmp = umxThresholdMatrix(x)

# ==================
# = Binary example =
# ==================
require(umx)
if(!packageVersion("OpenMx") > 2.5){message("Update OpenMx to get a new version of twinData")}
data(twinData)
# ===============================================================
# = Create a series of binary and ordinal columns to work with =
# ===============================================================
# Cut to form category of 80 \% obese subjects
selDVs = c("obese1", "obese2")
obesityLevels = c('normal', 'obese')
cutPoints <- quantile(twinData[, "bmi1"], probs = .2, na.rm = TRUE)
twinData$obese1 <- cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 <- cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
# Step 2: Make the ordinal variables into mxFactors
# this ensures ordered= TRUE + requires user to confirm levels
twinData[, selDVs] <- mxFactor(twinData[, selDVs], levels = obesityLevels)

# Repeat for three-level weight variable
selDVs = c("obeseTri1", "obeseTri2")
obesityLevels = c('normal', 'overweight', 'obese')
cutPoints <- quantile(twinData[, "bmi1"], probs = c(.4, .7), na.rm = TRUE)
twinData$obeseTri1 <- cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obeseTri2 <- cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData[, selDVs] <- mxFactor(twinData[, selDVs], levels = obesityLevels)

selDVs = c("obeseQuad1", "obeseQuad2")
obesityLevels = c('underWeight', 'normal', 'overweight', 'obese')
cutPoints <- quantile(twinData[, "bmi1"], probs = c(.25, .4, .7), na.rm = TRUE)
twinData$obeseQuad1 <- cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obeseQuad2 <- cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData[, selDVs] <- mxFactor(twinData[, selDVs], levels = obesityLevels)

# Example 1
selDVs = c("obese1", "obese2")
tmp = umxThresholdMatrix(twinData[,selDVs], sep = "", verbose = TRUE) # informative messages

# ======================================
# = Ordinal (n categories > 2) example =
# ======================================
selDVs = c("obeseTri1", "obeseTri2")
tmp = umxThresholdMatrix(twinData[,selDVs], sep = "", verbose = TRUE)

# ========================================================
# = Mix of all three kinds example (and a 4-level trait) =
# ========================================================

selDVs = umx_paste_names(c("bmi", "obese", "obeseTri", "obeseQuad"), "", 1:2)
tmp = umxThresholdMatrix(twinData[,selDVs], sep = "", verbose = TRUE)
str(tmp)

}
\references{
- \url{http://tbates.github.io}, \url{https://github.com/tbates/umx}
}
\seealso{
Other Advanced Model Building Functions: \code{\link{umxJiggle}},
  \code{\link{umxLabel}}, \code{\link{umxLatent}},
  \code{\link{umxRAM2Ordinal}}, \code{\link{umxValues}},
  \code{\link{umx_add_variances}},
  \code{\link{umx_fix_first_loadings}},
  \code{\link{umx_fix_latents}},
  \code{\link{umx_get_bracket_addresses}},
  \code{\link{umx_string_to_algebra}}, \code{\link{umx}}
}
