% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictMethod.R
\name{predict.gsmar}
\alias{predict.gsmar}
\title{Forecast GMAR, StMAR, or G-StMAR process}
\usage{
\method{predict}{gsmar}(
  object,
  ...,
  n_ahead,
  nsimu = 10000,
  pi = c(0.95, 0.8),
  pred_type = c("median", "mean", "cond_mean"),
  pi_type = c("two-sided", "upper", "lower", "none"),
  plotRes = TRUE,
  mix_weights = TRUE,
  nt
)
}
\arguments{
\item{object}{object of class \code{'gsmar'} created with function \code{fitGSMAR} or \code{GSMAR}.}

\item{...}{additional arguments passed to \code{grid} (ignored if \code{plot_res==FALSE}).}

\item{n_ahead}{a positive integer specifying how many steps in the future should be forecasted.}

\item{nsimu}{a positive integer specifying to how many simulations the forecast should be based on.}

\item{pi}{a numeric vector specifying confidence levels for the prediction intervals.}

\item{pred_type}{should the prediction be based on sample "median" or "mean"? Or should it
be one-step-ahead forecast based on the exact conditional mean (\code{"cond_mean"})? prediction
intervals won't be calculated if the exact conditional mean is used.}

\item{pi_type}{should the prediction intervals be "two-sided", "upper", or "lower"?}

\item{plotRes}{a logical argument defining whether the forecast should be plotted or not.}

\item{mix_weights}{\code{TRUE} if forecasts for mixing weights should be plotted, \code{FALSE} in not.}

\item{nt}{a positive integer specifying the number of observations to be plotted
along with the prediction. Default is \code{round(length(data)*0.15)}.}
}
\value{
Returns a class \code{'gsmarpred'} object containing, among the specifications,...
 \describe{
   \item{$pred}{Point forecasts}
   \item{$pred_ints}{Prediction intervals}
   \item{$mix_pred}{Point forecasts for mixing weights}
   \item{mix_pred_ints}{Individual prediction intervals for mixing weights, as \code{[, , m]}, m=1,..,M.}
 }
}
\description{
\code{predict.gsmar} forecasts the specified GMAR, StMAR, or G-StMAR process by using the given
 data to simulate its possible future values. For one-step forecasts using the exact formula for conditional
 mean is supported.
}
\details{
\code{predict.gsmar} uses the last \code{p} values of the data to simulate \code{nsimu}
 possible future values for each step-ahead. The point prediction is then obtained by calculating
 the sample median or mean for each step and the prediction intervals are obtained from the
 empirical fractiles.

 The function \code{simulateGSMAR} can also be used directly for quantile based forecasting.
}
\examples{
\donttest{
# GMAR model
fit12 <- fitGSMAR(data=logVIX, p=1, M=2, model="GMAR")
pred12 <- predict(fit12, n_ahead=10, pi=c(0.95, 0.8))
pred12

# Non-mixture StMAR model, upper prediction intervals
fit11t <- fitGSMAR(logVIX, 1, 1, model="StMAR", ncores=1, ncalls=1)
predict(fit11t, n_ahead=10, pi_type="upper", pi=0.9)

# G-StMAR model, no prediction intervals
fit12gs <- fitGSMAR(logVIX, 1, M=c(1, 1), model="G-StMAR")
pred12gs <- predict(fit12gs, n_ahead=10, pred_type="median",
 pi_type="none", plotRes=FALSE)
pred12gs
plot(pred12gs)

# Restricted GMAR model, one-step conditional mean prediction
fit12r <- fitGSMAR(logVIX, 1, 2, model="GMAR", restricted=TRUE)
pred12r <- predict(fit12r, pred_type="cond_mean", plotRes=FALSE)
pred12r

# Such StMAR(3,2) that the AR coefficients are restricted to be
# the same for both regimes and that the second AR coefficients are
# constrained to zero.
fit32rc <- fitGSMAR(logVIX, 3, 2, model="StMAR", restricted=TRUE,
 constraints=matrix(c(1, 0, 0, 0, 0, 1), ncol=2))
predict(fit32rc, n_ahead=3, pi_type="lower")
}
}
\references{
\itemize{
   \item Galbraith, R., Galbraith, J. 1974. On the inverses of some patterned matrices arising
           in the theory of stationary time series. \emph{Journal of Applied Probability} \strong{11}, 63-71.
   \item Kalliovirta L. (2012) Misspecification tests based on quantile residuals.
           \emph{The Econometrics Journal}, \strong{15}, 358-393.
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2015. Gaussian Mixture Autoregressive model for univariate time series.
           \emph{Journal of Time Series Analysis}, \strong{36}, 247-266.
   \item Meitz M., Preve D., Saikkonen P. 2018. A mixture autoregressive model based on Student's t-distribution.
           arXiv:1805.04010 \strong{[econ.EM]}.
   \item Virolainen S. 2020. A mixture autoregressive model based on Gaussian and Student's t-distribution.	arXiv:2003.05221 [econ.EM].
 }
}
\seealso{
\code{\link{simulateGSMAR}}, \code{\link{condMoments}}, \code{\link{fitGSMAR}}, \code{\link{GSMAR}},
 \code{\link{quantileResidualTests}}, \code{\link{diagnosticPlot}}
}
