# ***************************************************************************
# Copyright (C) 2016 Juergen Altfeld (R@altfeld-im.de)
# ---------------------------------------------------------------------------
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ***************************************************************************



#' Convert a call stack into a list of printable strings
#'
#' @description    Converts a call stack into a list of printable strings ("labels") with a limited length per call.
#'                 If source code references are available they are also added in the form
#'                 \<file name\>#\<line number\>: executed R expression (call)
#'
#' @param value    a list of calls ("call.stack") generated by \code{\link{sys.calls}}
#' @param compact  if TRUE only calls that contain a source code reference (attribute "srcref") are returned
#'                 (plus always the first call); if FALSE all calls will be returned.
#' @param maxwidth Maximum number of characters per call in the return value (longer strings will be cutted).
#'                 Must be between 40 and 2000 (until version 1.2.2: 1000)
#'
#' @return         A list of strings (one for each call).
#'                 If \code{compact} is \code{TRUE} at the last call is returned even if it does not contain
#'                 a source code reference.
#'
#' @details        R does track source code references only if you set the option "keep.source" to TRUE via
#'                 \code{options(keep.source = TRUE)}. Without this option this function cannot enrich source code references.
#'                 If you use \command{Rscript} to start a non-interactive R script as batch job you
#'                 have to set this option since it is FALSE by default. You can add this option to your
#'                 \link{.Rprofile} file or use a startup R script that sets this option and sources your
#'                 actual R script then.
#'
#'                 This function is based on the undocumented \code{\link{limitedLabels}} function of the base package.
#'                 The source code can be viewed by entering \code{limitedLabels} in the R console.
#'                 The attributes required to add source file names and line numbers to the calls (srcref and srcfile)
#'                 and how they are created internally are explained in this article:
#'                 \url{https://journal.r-project.org/archive/2010-2/RJournal_2010-2_Murdoch.pdf}
#'
#' @seealso        \code{\link{sys.calls}}, \code{\link{tryCatchLog}}, \code{\link{get.pretty.call.stack}}
#' @examples
#' limitedLabelsCompact(sys.calls(), TRUE)
#' @export
limitedLabelsCompact <- function(value, compact = FALSE, maxwidth = getOption("width") - 5L) {

  # create vector of source references (file and row numbers) for each call item of the stack
  srcrefs <- sapply(value, function(v) {

    srcref <- attr(v, "srcref")
    if (!is.null(srcref)) {
      srcfile <- attr(srcref, "srcfile")
      paste0(basename(srcfile$filename), "#", srcref[1L], ": ")
    }
    else ""
  })

  # create a list of only first line of code for each call stack item
  if (compact == TRUE)
    value <- lapply(as.character(value), function(x) strsplit(x, "\n")[[1]][1])


  # combine source references with the call stack and ident multi-line code
  value <- gsub("\n", "\n    ", as.character(value))
  value <- paste0(srcrefs, value)

  # cut lines that are too long
  if (is.null(maxwidth) || maxwidth < 40L)
    maxwidth <- 40L
  maxwidth <- min(maxwidth, 2000L)  # May 19, 2021: Maxwidth changed from 1000 to 2000 due to unit testing problem (issue #64)
  value <- strtrim(value, maxwidth)

  if (compact == TRUE) {
    # return only call stack items that contain a source reference
    srcrefs.available <- srcrefs != ""
    srcrefs.available[1] <- TRUE          # always return the first row!
    value <- value[srcrefs.available]
  }

  return(value)
}
