//
// CG-sparse.cpp.  Part of the trustOptim package for the R programming language.
//
// This file is part of trustOptim, a nonlinear optimization package
// for the R statistical programming platform.
//
// Copyright (C) 2012 Michael Braun
//
// This Source Code Form is subject to the license terms of the Mozilla
// Public License v. 2.0. If a copy of the MPL was not distributed
// with this file, you can obtain one at http://mozilla.org/MPL/2.0/.
// See the trustOptim LICENSE file for more information.


#ifndef __TRUST_OPTIM_CGSPARSE
#define __TRUST_OPTIM_CGSPARSE

#include <include_eigen.hpp>

#include <CG-base.h>

using Eigen::Dynamic;

template<typename TP, typename TFunc, typename THess, typename TPreLLt>  // TP is type for the parameter vector
  class Trust_CG_Sparse : public Trust_CG_Base<TP, TFunc, THess, TPreLLt> {


  typedef typename TP::Index Index;

  typedef SparseMatrix<double> SparseMatrixXd;

  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::Bk;
  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::nvars;
  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::func;
  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::function_scale_factor;
  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::xk;
  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::precond_ID;
  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::PrecondLLt;
  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::status;
  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::rad;
  using Trust_CG_Base<TP, TFunc, THess, TPreLLt>::iter;


private:


  // Tool to compute hessian.  Returns a dense representation


  Index nnz;

  void update_precond();
  void update_precond_diag();
  void update_precond_identity();
  void update_precond_modified_Cholesky();

  void init_precond();
  void init_precond_diag();
  void init_precond_identity();
  void init_precond_modified_Cholesky();

  void init_sparse_hessian();
  void update_sparse_hessian();

  void update_hessian();

public:


  // control values for the optimizer

  Trust_CG_Sparse(TFunc&, const MatrixBase<TP>&,	
		  const double &, const double &, const double &,
		  const double &, const int &, const int , const int &,
		  const int &, const double &,
		  const double &, const double &,
		  const double &, const double &, const double &, 
		  const int &, const int &, const int &);

 template<typename Tvec, typename Tout>
  MB_Status get_current_state(const MatrixBase<Tvec> &,
			      double &, const MatrixBase<Tvec> &,
			      const SparseMatrixBase<Tout> &,
			      int &, double &);

};

template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
  Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::Trust_CG_Sparse(TFunc & func_,
							      const MatrixBase<TP>& startX_,
							      const double & rad_,
							      const double & min_rad_,
							      const double & tol_,
							      const double & prec_,
							      const int & report_freq_,
							      const int  report_level_,
							      const int & report_precision_,
							      const int & maxit_,
							      const double & contract_factor_,
							      const double & expand_factor_,
							      const double & contract_threshold_,
							      const double & expand_threshold_rad_,
							      const double & expand_threshold_ap_,
							      const double & function_scale_factor_,
							      const int & precond_refresh_freq_,
							      const int & precond_ID_,
							      const int & trust_iter_) :
  Trust_CG_Base<TP, TFunc, THess, TPreLLt>(func_, startX_, rad_, min_rad_, tol_, prec_, report_freq_,
					   report_level_,
					   report_precision_, maxit_, contract_factor_,
					   expand_factor_, contract_threshold_,
					   expand_threshold_rad_, expand_threshold_ap_,
					   function_scale_factor_, precond_refresh_freq_, precond_ID_,
					   trust_iter_),
    nnz(func_.get_nnz())
  {
    
    init_sparse_hessian();
    Bk *= function_scale_factor;
    
    init_precond();     
  }

template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
  void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::init_sparse_hessian() {

  using std::endl;

  Bk.resize(nvars, nvars);
  Bk.reserve(nnz);
  
  func.get_hessian(xk, Bk);
}

template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::update_sparse_hessian() {

  func.get_hessian(xk, Bk);

}

template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::update_hessian() {

  update_sparse_hessian(); // get a brand new Hessian
  Bk *= function_scale_factor;

}

template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
template<typename Tvec, typename Tout>
  MB_Status Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::get_current_state(const MatrixBase<Tvec> & pars_,
									  double & fval,
									  const MatrixBase<Tvec> & grad_,
									  const SparseMatrixBase<Tout> & hess_,
									  int & iterations,
									  double & radius)
{

   MatrixBase<Tvec> & pars = const_cast<MatrixBase<Tvec>& >(pars_);
   MatrixBase<Tvec> & grad = const_cast<MatrixBase<Tvec>& >(grad_);
   SparseMatrixBase<Tout> & hess = const_cast<SparseMatrixBase<Tout>&>(hess_);

   pars = xk;
   func.get_fdf(xk, fval, grad);
   func.get_hessian(xk, hess);

   iterations = iter;
   radius = rad;

   return status;
}

template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::init_precond_diag() {

  // this is a special case of a diagonal preconditioner
  // in general, preconditioner must be lower triangular and positive definite

  SparseMatrixXd tmp(nvars, nvars);
  tmp.reserve(Eigen::VectorXi::Constant(nvars,1));

  double el;

  for (int j=0; j<nvars; j++) {
    el = Bk.coeff(j,j);
    tmp.insert(j,j) = el;
  }
   tmp.makeCompressed();

  PrecondLLt.analyzePattern(tmp);
  PrecondLLt.factorize(tmp);

  return;
}

template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::update_precond_diag() {

  // this is a special case of a diagonal preconditioner
 
  SparseMatrixXd tmp(nvars, nvars);
  tmp.reserve(Eigen::VectorXi::Constant(nvars,1));

  double el;
  for (int j=0; j<nvars; j++) {
    el = Bk.coeff(j,j);
    tmp.insert(j,j) = el;
  }
  tmp.makeCompressed();


  // PrecondLLt already analyzed
  PrecondLLt.factorize(tmp);

  return;
}


template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::init_precond_identity() {

  SparseMatrixXd tmp(nvars, nvars);
  tmp.reserve(nvars);

  for (int j=0; j<nvars; j++) {
    tmp.startVec(j);
    tmp.insertBack(j,j) = 1.;
  }
  tmp.makeCompressed();

  PrecondLLt.compute(tmp);

  return;


}


template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::update_precond_identity() {

  // this is a special case of a diagonal preconditioner
  
  // Do nothing.  Preconditioner will always be identity matrix
 
  return;
}

template<typename TP, typename TFunc, typename THess, typename TPreLLt>
void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::init_precond() {

  switch (precond_ID) {
  case 0:
    init_precond_identity();
    break;
  case 1:
    init_precond_diag();
    break;
  case 2:
    init_precond_modified_Cholesky();
    break;
  default:
    init_precond_identity();
  }

  return;
}


template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::update_precond() {

  switch (precond_ID) {

  case 0:
    update_precond_identity();
    break;
  case 1:
    update_precond_diag();
    break;
  case 2:
    update_precond_modified_Cholesky();
    break;
   default:
    update_precond_identity();

  }

  return;
}


template<typename TP, typename TFunc, typename THess, typename TPreLLt> 
void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::init_precond_modified_Cholesky() {

  // this is a special case of a hessian preconditioner
  // in general, preconditioner must be lower triangular and positive definite
 

  PrecondLLt.analyzePattern(Bk); // set sparsity pattern same as Hessian

  update_precond_modified_Cholesky();
}

template<typename TP, typename TFunc, typename THess, typename TPreLLt>
  void Trust_CG_Sparse<TP, TFunc, THess, TPreLLt>::update_precond_modified_Cholesky() {
 
  bool success = FALSE;
  double alpha, beta, bmin;

  VectorXd T(nvars);
  SparseMatrixXd BB(nvars, nvars);
  BB = Bk.template selfadjointView<Lower>();
 
  for (int j=0; j<BB.outerSize(); j++) {
    T(j) = sqrt(BB.innerVector(j).dot(BB.innerVector(j)));
  }

  for (int j=0; j<BB.outerSize(); j++) {
    for (typename SparseMatrixXd::InnerIterator it(BB, j); it; ++it) {
      BB.coeffRef(it.row(),j) *= 1/sqrt(T(it.row()) * T(j));
    }
  }

  beta = sqrt(BB.cwiseAbs2().sum());
  bmin = BB.coeff(0,0);
  for (int j=0; j<nvars; j++) {
    bmin = std::min(bmin,BB.coeff(j,j));
  }

  if (bmin > 0) alpha = 0; else alpha = beta/2;

  int ii = 0;
  do {
    ii++;
    
    PrecondLLt.factorize(BB); // try decomp of modified Hessian
    if (PrecondLLt.info() == Success) {
      success = TRUE;
    } else {
      alpha = std::max(2*alpha, beta/2.) - alpha;
      for (int j = 0; j<nvars; j++) {
	BB.coeffRef(j,j) += alpha;
      }
    }

  }
  while (!success);

}







#endif


