% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chemdose_toc.R
\name{chemdose_toc_once}
\alias{chemdose_toc_once}
\title{Apply `chemdose_toc` function and output a data frame}
\usage{
chemdose_toc_once(
  df,
  input_water = "defined_water",
  alum = 0,
  ferricchloride = 0,
  ferricsulfate = 0,
  coeff = "Alum"
)
}
\arguments{
\item{df}{a data frame containing a water class column, which has already been computed using
\code{\link{define_water_chain}}. The df may include a column named for the coagulant being dosed,
and a column named for the set of coefficients to use.}

\item{input_water}{name of the column of Water class data to be used as the input for this function. Default is "defined_water".}

\item{alum}{Hydrated aluminum sulfate Al2(SO4)3*14H2O + 6HCO3 -> 2Al(OH)3(am) +3SO4 + 14H2O + 6CO2}

\item{ferricchloride}{Ferric Chloride FeCl3 + 3HCO3 -> Fe(OH)3(am) + 3Cl + 3CO2}

\item{ferricsulfate}{Amount of ferric sulfate added in mg/L: Fe2(SO4)3*8.8H2O + 6HCO3 -> 2Fe(OH)3(am) + 3SO4 + 8.8H2O + 6CO2}

\item{coeff}{String specifying the Edwards coefficients to be used from "Alum", "Ferric", "General Alum", "General Ferric", or "Low DOC" or
named vector of coefficients, which must include: k1, k2, x1, x2, x3, b}
}
\value{
A data frame with an updated DOC, TOC, and UV254 concentration.
}
\description{
This function allows \code{\link{chemdose_toc}} to be added to a piped data frame.
Its output is a data frame with updated TOC, DOC, and UV254.
}
\details{
The data input comes from a `water` class column, as initialized in \code{\link{define_water}} or \code{\link{balance_ions}}.

If the input data frame has a column(s) name matching a valid coagulant(s), the function will dose that coagulant(s). Note:
The function can only dose a coagulant as either a column or from the function arguments, not both.

The column names must match the coagulant names as displayed in \code{\link{chemdose_toc}}.
To see which coagulants can be passed into the function, see \code{\link{chemdose_toc}}.

tidywater functions cannot be added after this function because they require a `water` class input.

 For large datasets, using `fn_once` or `fn_chain` may take many minutes to run. These types of functions use the furrr package
 for the option to use parallel processing and speed things up. To initialize parallel processing, use
 `plan(multisession)` or `plan(multicore)` (depending on your operating system) prior to your piped code with the
 `fn_once` or `fn_chain` functions. Note, parallel processing is best used when your code block takes more than a minute to run,
 shorter run times will not benefit from parallel processing.
}
\examples{

library(purrr)
library(furrr)
library(tidyr)
library(dplyr)

example_df <- water_df \%>\%
  define_water_chain() \%>\%
  balance_ions_chain() \%>\%
  chemdose_ph_chain(alum = 30) \%>\%
  chemdose_toc_once(input_water = "dosed_chem_water")

example_df <- water_df \%>\%
  define_water_chain() \%>\%
  balance_ions_chain() \%>\%
  mutate(
    ferricchloride = seq(1, 12, 1),
    coeff = "Ferric"
  ) \%>\%
  chemdose_toc_once(input_water = "balanced_water")

example_df <- water_df \%>\%
  define_water_chain() \%>\%
  balance_ions_chain() \%>\%
  chemdose_toc_once(input_water = "balanced_water", alum = 40, coeff = "General Alum")

# Initialize parallel processing
plan(multisession, workers = 2) # Remove the workers argument to use all available compute
example_df <- water_df \%>\%
  define_water_chain() \%>\%
  balance_ions_chain() \%>\%
  mutate(ferricchloride = seq(1, 12, 1)) \%>\%
  chemdose_toc_once(input_water = "balanced_water", coeff = "Ferric")

# Optional: explicitly close multisession processing
plan(sequential)

}
\seealso{
\code{\link{chemdose_toc}}
}
