% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutate.R
\name{mutate_}
\alias{mutate_}
\title{Drop-in replacement for \code{\link[dplyr]{mutate}}}
\usage{
mutate_(x, ..., .by, .order_by, .frame, .index, .desc = FALSE)
}
\arguments{
\item{x}{(data.frame)}

\item{...}{expressions to be passed to \code{\link[dplyr]{mutate}}}

\item{.by}{(character vector, optional: Yes) columns to group by}

\item{.order_by}{(character vector, optional: Yes) columns to order by}

\item{.frame}{(vector, optional: Yes) Vector of length 2 indicating the
number of rows to consider before and after the current row. When argument
\code{.index} is provided (typically a column of type date or datetime), before
and after can be
\href{https://lubridate.tidyverse.org/reference/interval.html}{interval}
objects. See examples.}

\item{.index}{(string, optional: Yes) name of index column}

\item{.desc}{(logical_vector, default: FALSE) bool or logical vector of same
length as \code{.order_by}.}
}
\value{
data.frame
}
\description{
Provides supercharged version of \code{\link[dplyr]{mutate}}
with \code{group_by}, \code{order_by} and aggregation over arbitrary window frame
around a row. This function allows some arguments to be passed as strings
instead of expressions.
}
\details{
A window function returns a value for every input row of a dataframe
based on a group of rows (frame) in the neighborhood of the input row. This
function implements computation over groups (\code{partition_by} in SQL) in a
predefined order (\code{order_by} in SQL) across a neighborhood of rows (frame)
defined by a (up, down) where
\itemize{
\item up/down are number of rows before and after the corresponding row
\item up/down are interval objects (ex: \code{c(days(2), days(1))})
}

This implementation is inspired by spark's \href{https://www.databricks.com/blog/2015/07/15/introducing-window-functions-in-spark-sql.html}{window API}.

Implementation Details:
\itemize{
\item Iteration per row over the window is implemented using the versatile
\href{https://cran.r-project.org/package=slider}{\code{slider}}.
\item Application of a window aggregation can be optionally run in parallel
over multiple groups (see argument \code{.by}) by setting a
\href{https://cran.r-project.org/package=future}{future} parallel backend. This
is implemented using \href{https://cran.r-project.org/package=furrr}{furrr}
package.
\item function subsumes regular usecases of \code{\link[dplyr]{mutate}}
}
}
\examples{
library("magrittr")
# example 1
# Using iris dataset,
# compute cumulative mean of column `Sepal.Length`
# ordered by `Petal.Width` and `Sepal.Width` columns
# grouped by `Petal.Length` column

iris \%>\%
  mutate_(sl_mean = mean(Sepal.Length),
          .order_by = c("Petal.Width", "Sepal.Width"),
          .by = "Petal.Length",
          .frame = c(Inf, 0),
          ) \%>\%
  dplyr::slice_min(n = 3, Petal.Width, by = Species)

# example 2
# Using a sample airquality dataset,
# compute mean temp over last seven days in the same month for every row

airquality \%>\%
  # create date column
  dplyr::mutate(date_col = as.Date(paste("1973",
                                         stringr::str_pad(Month,
                                                          width = 2,
                                                          side = "left",
                                                          pad = "0"
                                                          ),
                                         stringr::str_pad(Day,
                                                          width = 2,
                                                          side = "left",
                                                          pad = "0"
                                                          ),
                                         sep = "-"
                                         )
                                  )
                ) \%>\%
  # create gaps by removing some days
  dplyr::slice_sample(prop = 0.8) \%>\%
  # compute mean temperature over last seven days in the same month
  mutate_(avg_temp_over_last_week = mean(Temp, na.rm = TRUE),
          .order_by = "Day",
          .by = "Month",
          .frame = c(lubridate::days(7), # 7 days before current row
                     lubridate::days(-1) # do not include current row
                     ),
          .index = "date_col"
          )
}
\seealso{
mutate
}
