#' Create a test.
#' 
#' A test encapsulates a series of expectations about small, self-contained
#' set of functionality.  Each test is contained in a \link{context} and
#' contains multiple expectation generated by \code{\link{expect_that}}.  
#' 
#' Tests are evaluated in their own environments, and should not affect 
#' global state.
#' 
#' When run from the command line, tests return \code{NULL} if all 
#' expectations are met, otherwise it returns 
#'
#' @param desc test name.  Names should be kept as brief as possible, as they
#'   are often used as line prefixes.
#' @param code test code containing expectations
#' @examples
#' test_that("trigonometric functions match identies", {
#'   expect_that(sin(pi / 4), equals(1 / sqrt(2)))
#'   expect_that(cos(pi / 4), equals(1 / sqrt(2)))
#'   expect_that(tan(pi / 4), equals(1))
#' })
#' # Failing test:
#' \dontrun{
#' test_that("trigonometric functions match identies", {
#'   expect_that(sin(pi / 4), equals(1))
#' })
#' }
test_that <- function(desc, code) {
  test_reporter()$start_test(desc)
  
  env <- new.env(parent = globalenv())  
  res <- try_capture_stack(substitute(code), env)
  
  if (inherits(res, "error")) {
    traceback <- create_traceback(res$calls)
    report <- error_report(res, traceback)
    test_reporter()$add_result(report)
  }
  
  test_reporter()$end_test()
}

#' Generate error report from traceback.
#'
#' @keywords internal
#' @param error error message
#' @param traceback traceback generated by \code{\link{create_traceback}}
error_report <- function(error, traceback) {
  msg <- gsub("Error.*?: ", "", as.character(error))
  
  if (length(traceback) > 0) {
    user_calls <- paste(traceback, collapse = "\n")      
    msg <- paste(msg, user_calls, sep = "")
  } else {
    # Need to remove trailing newline from error message to be consistent
    # with other messages
    msg <- gsub("\n$", "", msg)
  }
  
  expectation(NA, msg)
}

#' Generate a traceback from a list of calls.
#' 
#' @param callstack stack of calls, as generated by (e.g.) 
#'   \code{\link[base]{sys.calls}}
#' @keywords internal
create_traceback <- function(callstack) {
  calls <- lapply(callstack, deparse)
  calls <- sapply(calls, paste, collapse = "\n")
  first_eval <- match("eval(expr, envir, enclos)", calls, 0)[1]
  
  if (first_eval == length(calls)) return()
  
  user_calls <- calls[-seq_len(first_eval)]
  user_calls <- paste(seq_along(user_calls), ": ", user_calls, sep = "")
  user_calls <- gsub("\n", "\n   ", user_calls)
  user_calls
  
}

#' Try, capture stack on error.
#'
#' This is a variant of \code{\link{tryCatch}} that also captures the call
#' stack if an error occurs.
#'
#' @param quoted_code code to evaluate, in quoted form
#' @param env environment in which to execute code
#' @keywords internal
try_capture_stack <- function(quoted_code, env) {
  capture_calls <- function(e) {
    # Capture call stack, removing last two calls, which are added by
    # withCallingHandlers
    e$calls <- head(sys.calls(), -2)
    signalCondition(e)
  }
  
  tryCatch(
    withCallingHandlers(eval(quoted_code, env), error = capture_calls),
    error = identity
  )
}