% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/updateModelTerms.R
\name{updateModelTermInputs}
\alias{updateModelTermInputs}
\title{Methods for Computing and Updating ERGM/STERGM Term Inputs}
\usage{
updateModelTermInputs(dat, network = 1)
}
\arguments{
\item{dat}{EpiModel dat object tracking simulation state}

\item{network}{Numberic number of network location for multi-network simulations.}
}
\value{
Returns an updated \code{dat} object with the network list structure inputs
used by \code{\link{simulate_network}} or \code{\link{simulate_ergm}} with changes
to network size or nodal covariates.
}
\description{
Function to appropriately update model inputs based on ERGM model
             terms when using networkLite representation.
}
\details{
Calls \code{ergm_model} to update model inputs based on potential
exogenous changes to network structure (e.g., number of nodes) or nodal attributes
used within ERGM model (see example below). This function is typically used
within \code{EpiModel} module for network resimulation, immediately prior to
calling \code{\link{simulate_network}} or \code{\link{simulate_ergm}}.

Implemented terms are:
 \itemize{
   \item \code{edges}
   \item \code{nodematch}
   \item \code{nodefactor}
   \item \code{concurrent} (including heterogenous by attribute)
   \item \code{degree} (including heterogenous by attribute)
   \item \code{degrange}
   \item \code{absdiff}
   \item \code{absdiffby} (in the EpiModel package)
   \item \code{nodecov}
   \item \code{nodemix}
   \item \code{absdiffnodemix} (in the EpiModel package)
   \item \code{triangle}
   \item \code{gwesp(fixed=TRUE)}
 }
All other ERGM terms will return errors.
}
\examples{
\dontrun{
library("EpiModel")

# Set seed for reproducibility
set.seed(1234)

nw <- network_initialize(100)
nw <- set_vertex_attribute(nw, "group", rep(1:2, each = 50))
formation <- ~edges + nodefactor("group")
target.stats <- c(15, 10)
coef.diss <- dissolution_coefs(dissolution = ~offset(edges), duration = 1)
x <- netest(nw, formation, target.stats, coef.diss, verbose = FALSE)

param <- param.net(inf.prob = 0.3, inf.prob.g2 = 0.1)
init <- init.net(i.num = 10, i.num.g2 = 10)
control <- control.net(type = "SI", nsteps = 100, nsims = 5, tergmLite = TRUE)

# Full network structure after initialization
dat <- crosscheck.net(x, param, init, control)
dat <- initialize.net(x, param, init, control)
str(dat, max.level = 1)

# Examine the network list structure for nodefactor term
dat$p[[1]]$model.form$terms[[2]]

# inputs vector corresponds to group attribute stored here
dat$attr$group

# As example of what could happen in EpiModel: randomly reshuffle group
#   attribute values of 100 nodes
dat$attr$group <- sample(dat$attr$group)
dat$attr$group

# Update network list structure
dat <- updateModelTermInputs(dat)

# Check that network list structure for nodefactor term has been updated
dat$p[[1]]$model.form$terms[[2]]
}

}
