% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tab.R
\name{tab}
\alias{tab}
\title{Single cross-table, with color helpers}
\usage{
tab(
  data,
  row_var,
  col_var,
  tab_vars,
  wt,
  sup_cols,
  na = "keep",
  digits = 0,
  pct = "no",
  color = "no",
  diff = "tot",
  comp = "tab",
  totaltab = "line",
  totaltab_name = "Ensemble",
  tot = c("row", "col"),
  total_names = "Total",
  chi2 = NULL,
  ci = NULL,
  conf_level = 0.95,
  ci_visible = NULL,
  subtext = "",
  cleannames = NULL,
  rare_to_other = FALSE,
  n_min = 30,
  other_level = "Others",
  filter
)
}
\arguments{
\item{data}{A data frame.}

\item{row_var, col_var}{The row variable, which will be printed with one level per line,
and the column variable, which will be printed with one level per column.
For numeric variables means are calculated, in a single column.}

\item{tab_vars}{<\link[tidyr:tidyr_tidy_select]{tidy-select}> Tab variables :
a subtable is made for each combination of levels of the selected variables.
Leave empty to make a simple cross-table. All \code{tab_vars} are converted to factor.}

\item{wt}{A weight variable, of class numeric. Leave empty for unweighted results.}

\item{sup_cols}{<\link[tidyr:tidyr_tidy_select]{tidy-select}>
Supplementary columns variables, with only the first level printed, and row percentages
(for numeric variables, a mean will be calculated for each \code{row_var}).
To pass many variables you may use syntax \code{sup_cols = c(sup_col1, sup_col2, ...)}.
To keep all levels of other \code{col_vars}, or other types of percentages,
use \code{\link{tab_many}} instead.}

\item{na}{The policy to adopt with missing values, as a single string (for a more
precise control over the behavior of \code{NA}'s, vectorized for each variable,
use \code{\link{tab_many}}).
\itemize{
\item \code{"keep"}: by default, \code{NA}'s of row, col and tab variables are printed
as explicit "NA" level. Observations with NA in \code{sup_cols} variables are always
kept to calculate the base table, always removed to calculate supplementary cols.
\item \code{"drop"}: removes NA of row, col and tab variables.
}}

\item{digits}{The number of digits to print, as a single integer. To print a different
number of digits for each \code{sup_cols}, an integer vector of length
1 + \code{sup_cols} (the first being the number of digits for the base table).}

\item{pct}{The type of percentages to calculate, passed to \code{\link{tab_pct}} :
\itemize{
\item \code{"row"}: row percentages.
\item \code{"col"}: column percentages.
\item \code{"all"}: frequencies for each subtable/group, if there is \code{tab_vars}.
\item \code{"all_tabs"}: frequencies for the whole (set of) table(s).
}}

\item{color}{The type of colors to print, as a single string :
\itemize{
\item \code{"no"}: by default, no colors are printed.
\item \code{"diff"}: color percentages and means based on cells differences from
totals (or from first cells when \code{diff = "first"}).
\item \code{"diff_ci"}: color pct and means based on cells differences from totals
or first cells, removing coloring when the confidence interval of this difference
is higher than the difference itself.
\item \code{"after_ci"}: idem, but cut off the confidence interval from the
difference first.
\item \code{"contrib"}: color cells based on their contribution to variance
(except mean columns, from numeric variables).
\item \code{"auto"}: frequencies (\code{pct = "all"}, \code{pct = "all_tabs"})
and counts are colored with \code{"contrib"}.
When \code{ci = "diff"}, row and col percentages are colored with "after_ci" ;
otherwise they are colored with "diff".
}}

\item{diff}{The reference cell to calculate differences (used to print \code{colors}) :
\itemize{
\item \code{"tot"}: by default, cells differences from total rows are calculated with
\code{pct = "row"}, and cells differences from total columns with \code{pct = "col"}.
\item \code{"first"}: calculate cells differences from the first cell
of the row or column (useful to color temporal developments).
\item \code{"no"}: not use diffs to gain calculation time.
}}

\item{comp}{The comparison level : by subtables/groups, or for the whole table.
\itemize{
\item \code{"tab"}: by default, contributions to variance,
row differences from totals/first cells, and row confidence intervals for these
differences, are calculated for each \code{tab_vars} group.
\item \code{"all"}: compare cells to the general total line (provided there is
a total table with a total row), or with the first line of the total table
when \code{diff = "first"}.
}}

\item{totaltab}{The total table, to create with \code{\link{tab_totaltab}},
if there are subtables/groups (i.e. when \code{tab_vars} is provided) :
\itemize{
\item \code{"line"}: by default, add a general total line (necessary for
calculations with \code{comp = "all"})
\item \code{"table"}: add a complete total table
(i.e. \code{row_var} by \code{col_vars} without \code{tab_vars}).
\item \code{"no"}: not to draw any total table.
}}

\item{totaltab_name}{The name of the total table, as a single string.}

\item{tot}{The totals, to create with \code{\link{tab_tot}} :
\itemize{
\item \code{c("col", "row")} or \code{"both"} : by default, both total rows and total
columns.
\item \code{"row"}: only total rows.
\item \code{"col"}: only total column.
\item \code{"no"}: remove all totals (after calculations if needed).
}}

\item{total_names}{The names of the totals, as a character vector of length one or two.
Use syntax of type \code{c("Total row", "Total column")} to set different names for
rows and cols.}

\item{chi2}{Set to \code{TRUE} to calculate Chi2 summaries with \code{\link{tab_chi2}}.
Useful to print metadata, and to color cells based on their contribution to variance
(\code{color = "contrib"}). Automatically added if needed for \code{color}.}

\item{ci}{The type of confidence intervals to calculate, passed to \code{\link{tab_ci}}
(automatically added if needed for \code{color}).
\itemize{
\item \code{"cell"}: absolute confidence intervals of cells percentages.
\item \code{"diff"}: confidence intervals of the difference between a cell and the
relative total cell (or relative first cell when \code{diff = "first"}).
\item \code{"auto"}: \code{ci = "diff"} for means and row/col percentages,
\code{ci = "cell"} for frequencies ("all", "all_tabs").
}
By default, for percentages, with \code{ci = "cell"} Wilson's method is used,
and with \code{ci = "diff"} Wald's method along Agresti and Caffo's adjustment.
Means use classic method. This can be changed in \code{\link{tab_ci}}.}

\item{conf_level}{The confidence level, as a single numeric between 0 and 1.
Default to 0.95 (95\%).}

\item{ci_visible}{By default, confidence intervals are calculated and used to set
colors, but not printed. Set to \code{TRUE} to print them in the result.}

\item{subtext}{A character vector to print rows of legend under the table.}

\item{cleannames}{Set to \code{TRUE} to clean levels names, by removing
prefix numbers like "1-", and text in parenthesis. All data formatting arguments are
passed to \code{\link{tab_prepare}}.}

\item{rare_to_other}{When set to \code{TRUE}, levels with less count
than \code{n_min} will be merged into an "Other" level.}

\item{n_min}{The count under which a level is aggregated in the "Other" level.}

\item{other_level}{The name of the "Other" level, as a single string.}

\item{filter}{A \code{\link[dplyr:filter]{dplyr::filter}} to apply to the data frame
first, as a single string (which will be converted to code, i.e. to a call).
Useful when printing multiples tabs with \code{\link[tibble:tribble]{tibble::tribble}},
to use different filters for similar tables or simply make the field of observation
more visible into the code.}
}
\value{
A \code{tibble} of class \code{tab}, possibly with colored reading helpers.
All non-text columns are of class \code{\link{fmt}}, storing all
the data necessary to print formats and colors. Columns with \code{row_var} and
\code{tab_vars} are of class \code{factor} : every added \code{factor} will be
considered as a \code{tab_vars} and used for grouping. To add text columns without
using them in calculations, be sure they are of class \code{character}.
}
\description{
A full-featured function to create, manipulate and format single
cross-tables, using colors to make the printed tab more easily readable
(in R terminal or exported to Excel with \code{\link{tab_xl}}).
Since objects of class \code{tab} are also of class \code{tibble}, you can then use all
\pkg{dplyr} verbs to modify the result, like \code{\link[dplyr:select]{select}},
like \code{\link[dplyr:arrange]{arrange}}, \code{\link[dplyr:filter]{filter}}
or \code{\link[dplyr:mutate]{mutate}}.
Wrapper around the more powerful \code{\link{tab_many}}.
}
\examples{
# A simple cross-table:
tab(forcats::gss_cat, marital, race)

# With one numeric row or col variables it calculates means by category:
tab(forcats::gss_cat, marital, age)

# With more variables provided, `tab` makes a subtables for each combination of levels:
\donttest{
tab(forcats::gss_cat, marital, tab_vars = c(year, race))
}

# You can also add supplementary columns, text or numeric:
\donttest{
tab(dplyr::storms, category, status, sup_cols = c("pressure", "wind"))
}

# Colors to help the user read the table:
data <- forcats::gss_cat \%>\%
  dplyr::filter(year \%in\% c(2000, 2006, 2012), !marital \%in\% c("No answer", "Widowed"))
gss  <- "Source: General social survey 2000-2014"
gss2 <- "Source: General social survey 2000, 2006 and 2012"

# Differences between the cell and it's subtable's total cell:
\donttest{
tab(data, race, marital, year, subtext = gss2, pct = "row", color = "diff")
}

# Differences between the cell and the whole table's general total cell:
\donttest{
tab(data, race, marital, year, subtext = gss2, pct = "row", color = "diff",
  comp = "all")
}

# Historical differences:
\donttest{
data2 <- data \%>\% dplyr::mutate(year = as.factor(year))
tab(data2, year, marital, race, subtext = gss2, pct = "row",
    color = "diff", diff = "first", tot = "col")


# Differences with the total, except if their confidences intervals are superior to them:
tab(forcats::gss_cat, race, marital, subtext = gss, pct = "row", color = "diff_ci")

# Same differences, minus their confidence intervals:
tab(forcats::gss_cat, race, marital, subtext = gss, pct = "row", color = "after_ci")

# Contribution of cells to table's variance, like in a correspondence analysis:
tab(forcats::gss_cat, race, marital, subtext = gss, color = "contrib")
}

# Since the result is a tibble, you can use all dplyr verbs to modify it :
\donttest{
library(dplyr)
tab(dplyr::storms, category, status, sup_cols = c("pressure", "wind")) \%>\%
  dplyr::filter(category != "-1") \%>\%
  dplyr::select(-`tropical depression`) \%>\%
  dplyr::arrange(is_totrow(.), desc(category))
}

\donttest{
# With `dplyr::arrange`, don't forget to keep the order of tab variables and total rows:
tab(data, race, marital, year, pct = "row") \%>\%
  dplyr::arrange(year, is_totrow(.), desc(Married))
  }
}
