#' Display a percentage.
#'
#' The \code{progress_percent} widget displays a text description of a percentage
#' value between 0% and 100%.
#'
#' @section Usage:
#'    \preformatted{progress_percent(eavesdrop, caption = "", maximum = 100,
#'    size = 1, placeOnGrid = c(1, 1), updates = 100, delay = 0, honest = FALSE,
#'    closeAtMaximum = FALSE)}
#'
#' @param eavesdrop The variable to track with a progress bar.
#' @param caption A small text string describing the percent value context.
#' @param maximum The maximum value of \code{eavesdrop} that marks the end of
#'    what to progress/track.
#' @param size A number used to designate the size (magnification) of the
#'    widget. The default is set to \code{1} which is 80 by 80 pixels. For
#'    example, setting to \code{3} will results in a widget 3-times the default
#'    size (240 by 240 pixels) and will occupy a grid area of 3 by 3.
#' @param placeOnGrid A row by column coordinate (e.g., \code{c(row-number, column-number)})
#'    of a grid that designates the position to draw the widget on the
#'    \code{switchboard}. Use \code{showGrid()} to help organize widget placement
#'    on dashboard. The default places the first widget in pipe chain to
#'    the \code{c(1, 1)} position, and all following on the same row stacked to
#'    the right.
#' @param updates The number of times the widget is to be updated (e.g., when
#'     it be modified/changed). The default updates the widget 100 times. Increase
#'     number for smoother progress bar animation.
#' @param delay Pause each update of the switchboard. Default has no delay,
#'    values are in seconds (e.g., \code{delay = 0.01} results in 0.01 second
#'    delay with each iteration).
#' @param honest When \code{TRUE}, it updates the widget by the true progression value.
#'     The default (\code{FALSE}) has a cosmetic modification to the progression
#'    value that helps update it in a prettier way.
#' @param closeAtMaximum Functions like \code{switchboard_close()} by closing
#'      the switchboard window when the eavesdropped value equals maximum. NOTE:
#'      if a widget has \code{closeAtMaximum = TRUE}, then this widget MUST
#'      be placed at the end (i.e., last widget) of the pipe chain.
#'
#' @return Nothing.
#'
#' @examples \dontrun{
#'      
#'      for (i in 1:250) {
#'        switchboard(delay = 0.01) %>%
#'          progress_percent(i, maximum = 250)
#'      }
#'      switchboard_close()
#'
#' }
#'
#' @family progress bars
#' @name progress_percent
NULL


#' @inheritParams progress_percent
#' @import tcltk magrittr
#' @export
progress_percent <- function(.switchboard, ...) {
  switchboard_engine(.switchboard,
                     constructor = progress_percent_construct(.switchboard, ...),
                     updater = progress_percent_update(.switchboard, ...), ...)
}


#' helper function that constructs canvas items of a progress_image widget
#' @keywords internal
progress_percent_construct <- function(.switchboard,
                                       label = " ",
                                       size = 1,
                                       ...) {

  aCanvas <- tcltk::tkcanvas(.switchboard, width = 80 * size, height = 80 * size, background = switchboard.env$mainColors[2], borderwidth = 0, highlightthickness = 0)
  tcltk::tkcreate(aCanvas, "text", 65 * size, 60 * size, text = "0", anchor = "se", font = paste(switchboard.env$font, floor(38 * size / 1.2)), fill = switchboard.env$mainColors[4], tags = c("percent", "percentColor"))
  tcltk::tkcreate(aCanvas, "text", 65 * size + 10 * size, 53 * size, text = "%", anchor = "se", font = paste(switchboard.env$font, floor(11 * size / 1.2)), fill = switchboard.env$mainColors[4], tags = "percentColor")
  tcltk::tkcreate(aCanvas, "text", 65 * size + 10 * size, 64 * size, text = paste(label), anchor = "se", font = paste(switchboard.env$font, floor(10 * size / 1.2)), fill = switchboard.env$mainColors[4])

  return(aCanvas)
}


#' helper function that updates the canvas items of a progress_image widget
#' @keywords internal
# TODO: label is presented as i when argument "label" no called by user
progress_percent_update <- function(.switchboard,
                                    eavesdrop = NULL,
                                    maximum = 100,
                                    size = 1,
                                    updates = 100,
                                    honest = FALSE,
                                    closeAtMaximum = FALSE,
                                    ...) {

  if(eavesdrop %% ceiling(maximum / updates) == 0) {
    aCanvas <- paste0(.switchboard$theSwitchboard, ".", .switchboard$theWidget)
    if(honest == FALSE) percentDisplayed <- animate(eavesdrop/maximum, maximum/maximum)
    else percentDisplayed <- eavesdrop/maximum
    finalPercent <- ceiling(percentDisplayed * 100)
    tcltk::tkitemconfigure(aCanvas, "percent", "-text", finalPercent)
    tcltk::tkitemconfigure(aCanvas, "percentColor", "-fill", switchboard.env$rangeColorsAlt[100 - finalPercent + 1])
  }
  if((closeAtMaximum == TRUE) && (eavesdrop == maximum)) switchboard_close()
}

