% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weighted_survey_scheme.R
\name{weighted_survey_scheme}
\alias{weighted_survey_scheme}
\title{Weighted survey scheme}
\usage{
weighted_survey_scheme(
  site_data,
  cost_column,
  survey_budget,
  weight_column,
  locked_in_column = NULL,
  locked_out_column = NULL,
  solver = "auto",
  verbose = FALSE
)
}
\arguments{
\item{site_data}{\code{\link[sf:sf]{sf::sf()}} object containing the candidate survey
sites.}

\item{cost_column}{\code{character} name of the column in the argument to
the argument to \code{site_data} that contains the cost for surveying each
site. No missing (\code{NA}) values are permitted.}

\item{survey_budget}{\code{numeric} vector of maximum budgets for the survey
schemes. No missing (\code{NA}) values are permitted.}

\item{weight_column}{\code{character} name of the column in the argument
to \code{site_data} with the weights for each site.}

\item{locked_in_column}{\code{character} (optional) name of the column in
the argument to \code{site_data} that contains \code{logical}
(\code{TRUE}/ \code{FALSE}) values indicating if certain sites should be
locked into the survey scheme.
No missing (\code{NA}) values are permitted.
Defaults to \code{NULL} such that no sites are locked in.}

\item{locked_out_column}{\code{character} (optional) name of the column in
the argument to \code{site_data} that contains \code{logical}
(\code{TRUE}/ \code{FALSE}) values indicating if certain sites should be
locked out of the survey scheme.
No missing (\code{NA}) values are permitted.
Defaults to \code{NULL} such that no sites are locked out.}

\item{solver}{\code{character} name of the optimization solver to use
for generating survey schemes.
Available options include: \code{"Rsymphony"}, \code{"gurobi"} and \code{"auto"}.
The \code{"auto"} method will use the Gurobi optimization software if
it is available; otherwise, it will use the SYMPHONY software
via the \pkg{Rsymphony} package.
Defaults to \code{"auto"}.}

\item{verbose}{\code{logical} indicating if information should be
printed while generating survey scheme(s). Defaults to \code{FALSE}.}
}
\value{
A \code{matrix} of \code{logical} (\code{TRUE}/ \code{FALSE})
values indicating if a site is selected in a scheme or not. Columns
correspond to sites, and rows correspond to different schemes.
}
\description{
Generate a survey scheme by selecting the set of sites with the greatest
overall weight value, a maximum budget for the survey scheme.
}
\details{
Let \eqn{J} denote the set of sites (indexed by \eqn{j}), and let
\eqn{b} denote the maximum budget available for surveying the sites.
Next, let \eqn{c_j} represent the cost of surveying each site
\eqn{j \in J}, and \eqn{w_j} denote the relative value (weight) for
surveying each site \eqn{j \in J}. The set of sites with the greatest
overall weight values, subject to a given budget can the be identified by
solving the following integer programming problem. Here,
\eqn{x_j} is the binary decision variable indicating each if site
is selected in the survey scheme or not.

\deqn{\mathit{Maximize} \space \sum_{j \in J} x_j w_i \\
  \mathit{subject \space to} \\
  \sum_{j \in J} x_j c_j \leq b}{
  Minimize sum_j^J (xi * wi) subject to sum_j^J (xi * ci) <= b}
}
\section{Solver}{

This function can use the \pkg{Rsymphony} package and
the \href{https://www.gurobi.com/}{Gurobi optimization software} to generate
survey schemes. Although the \pkg{Rsymphony} package
is easier to install because it is freely available on the
The Comprehensive R Archive Network (CRAN), it is strongly recommended to
install the \href{https://www.gurobi.com/}{Gurobi optimization software} and the
\pkg{gurobi} R package because it can generate survey schemes much faster.
Note that special academic licenses are available at no cost.
Installation instructions are available online for \href{https://www.gurobi.com/documentation/9.1/quickstart_linux/r_ins_the_r_package.html}{Linux}, \href{https://www.gurobi.com/documentation/9.1/quickstart_windows/r_ins_the_r_package.html}{Windows}, and \href{https://www.gurobi.com/documentation/9.1/quickstart_mac/r_ins_the_r_package.html}{Mac OS} operating systems.
}

\examples{
# set seed for reproducibility
set.seed(123)

# simulate data
x <- sf::st_as_sf(
  tibble::tibble(x = rnorm(4), y = rnorm(4),
                 w = c(0.01, 10, 8, 1),
                 cost = c(1, 1, 1, 1)),
  coords = c("x", "y"))

# plot site' locations and color by weight values
plot(x[, "w"], pch = 16, cex = 3)

# generate scheme without any sites locked in
s <- weighted_survey_scheme(x, cost_column = "cost", survey_budget = 2,
                             weight_column = "w")

# print solution
print(s)

# plot solution
x$s <- c(s)
plot(x[, "s"], pch = 16, cex = 3)
}
