% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GetKLEst.R
\name{GetKLEst}
\alias{GetKLEst}
\title{Retrieve the surface of Kullback-Leibler (KL) estimators}
\usage{
GetKLEst(fit, burnin = floor(fit$L/10), fixLS = FALSE, approx = FALSE,
  segment = 50)
}
\arguments{
\item{fit}{Object of class \code{damcmc_res} or \code{bdmcmc_res}.}

\item{burnin}{Number of initial realizations to discard. By default, it is 1/10 of the total number of iterations.}

\item{fixLS}{Logical requesting to check and fix label switching (if present). Default is FALSE.}

\item{approx}{Logical flag to request use of the identifiability constraint
to permute all realizations. Same parameter as in
function \code{\link{FixLS_da}}.}

\item{segment}{Number of segments to split the posterior realizations into. Each portion of posterior
realizations is used to calculate a single Kullback-Leibler realization. The KL estimator is the average of
all the KL realizations. Default is 50.}
}
\value{
An object of class \code{intensity_surface}.
}
\description{
This function calculates the Kullback-Leibler
estimators of the parameters of the components of the
mixture intensity, based on a DAMCMC or BDMCMC fit. This is a decision
theoretic estimator of the parameters, meaning that, we compute the
Posterior Expected Loss (PEL) using the KL loss function and then
find the parameter values that minimize the PEL.

For examples see

\url{http://faculty.missouri.edu/~micheasa/sppmix/sppmix_all_examples.html
#GetKLEst}
}
\examples{
\donttest{
#generate a surface
truemix_surf <- rmixsurf(m = 3, lambda=100, xlim = c(-3,3), ylim = c(-3,3))
plot(truemix_surf,main="True IPPP intensity surface")
#generate a point pattern
genPPP=rsppmix(intsurf = truemix_surf, truncate = FALSE)
#fit the IPPP model using DAMCMC
fit = est_mix_damcmc(genPPP, m = 3,L=20000)
#get the surfaces of posterior means, MAP and KL estimates
Meansest=GetPMEst(fit)
MAPest=GetMAPEst(fit)
KLest=GetKLEst(fit)
#plot all fitted surfaces
plot(Meansest,main="IPPP intensity surface of posterior means")
plot(MAPest,main="IPPP intensity surface of MAP estimates")
plot(KLest,main="IPPP intensity surface of KL estimates")
#fix labels (if label switching is detected)
KLestLSFixed=GetKLEst(fit,fixLS=TRUE,approx=FALSE)
plot(KLestLSFixed,main="IPPP intensity surface of KL estimates (LS fixed)")
#compare the four estimates against the truth
CompareSurfs(truemix_surf, Meansest, LL = 100, truncate = FALSE)
CompareSurfs(truemix_surf, MAPest, LL = 100, truncate = FALSE)
CompareSurfs(truemix_surf, KLest, LL = 100, truncate = FALSE)
CompareSurfs(truemix_surf, KLestLSFixed, LL = 100, truncate = FALSE)}

}
\seealso{
\code{\link{rmixsurf}},
\code{\link{rsppmix}},
\code{\link{est_mix_damcmc}},
\code{\link{GetPMEst}},
\code{\link{GetMAPEst}},
\code{\link{CompareSurfs}}
}
\author{
Jiaxun Chen, Sakis Micheas
}
