#
#
#   rmhmodel.R
#
#   $Revision: 1.32 $  $Date: 2009/10/16 20:33:51 $
#
#

rmhmodel <- function(...) {
  UseMethod("rmhmodel")
}

rmhmodel.rmhmodel <- function(model, ...) {
  return(model)
}

rmhmodel.list <- function(model, ...) {
  argnames <- c("cif","par","w","trend","types")
  ok <- argnames %in% names(model)
  do.call("rmhmodel.default", model[argnames[ok]])
}

rmhmodel.default <- function(...,
                     cif=NULL, par=NULL, w=NULL, trend=NULL, types=NULL)
{
  if(length(list(...)) > 0)
    stop(paste("Syntax should be rmhmodel(cif, par, w, trend, types)\n",
               "with arguments given by name if they are present"))
  
  # Validate parameters
  if(is.null(cif)) stop("cif is missing or NULL")
  if(is.null(par)) stop("par is missing or NULL")

  if(!is.null(w))
    w <- as.owin(w)
  
  if(!is.character(cif) || length(cif) != 1)
    stop("cif should be a character string")

  # Check that this is a recognised model
  # and look up the rules for this model
  rules <- .Spatstat.RmhTable[[cif]]
  if(is.null(rules))
    stop(paste("Unrecognised cif:", sQuote(cif)))
  
  # Map the name of the cif from R to C
  #      (the names are normally identical in R and C,
  #      except "poisson" -> NA)
  C.id <- rules$C.id
  
  # Check that the C name is recognised in C 
  if(!is.na(C.id)) {
    z <- .C("knownCif",
            cifname=as.character(C.id),
            answer=as.integer(0),
            PACKAGE="spatstat")
    ok <- as.logical(z$answer)
    if(!ok)
      stop(paste("Internal error: the cif", sQuote(C.id),
                 "is not recognised in the C code"))
  }

  # Validate the model parameters and reformat them 
  check <- rules$parhandler
  C.par <-
    if(!rules$multitype)
      check(par)
    else if(!is.null(types))
      check(par, types)
    else 
      # types vector not given - defer checking
      NULL

  # ensure it's a numeric vector
  C.par <- unlist(C.par)

  # Calculate reach of model
  mreach <- rules$reach(par)

###################################################################
# return augmented list  
  out <- list(cif=cif,
              par=par,
              w=w,
              trend=trend,
              types=types,
              C.id=C.id,
              C.par=C.par,
              check= if(is.null(C.par)) check else NULL,
              multitype.interact=rules$multitype,
              reach=mreach
              )
  class(out) <- c("rmhmodel", class(out))
  return(out)
}

print.rmhmodel <- function(x, ...) {
  verifyclass(x, "rmhmodel")

  cat("Metropolis-Hastings algorithm, model parameters\n")

  cat(paste("Conditional intensity: cif=", x$cif, "\n"))

  if(!is.null(x$types)) {
    if(length(x$types) == 1)
      cat("Univariate process.\n")
    else {
      cat("Multitype process with types =\n")
      print(x$types)
      if(!x$multitype.interact)
        cat("Interaction does not depend on type\n")
    }
  } else if(x$multitype.interact) 
    cat("Multitype process, types not yet specified.\n")
  
  cat("Numerical parameters: par =\n")
  print(x$par)
  if(is.null(x$C.par))
    cat("Parameters have not yet been checked for compatibility with types.\n")
  if(is.owin(x$w)) print(x$w) else cat("Window: not specified.\n")
  cat("Trend: ")
  if(!is.null(x$trend)) print(x$trend) else cat("none.\n")

}

reach.rmhmodel <- function(x, ...) {
  if(length(list(...)) == 0)
    return(x$reach)
  rules <- .Spatstat.RmhTable[[x$cif]]
  return(rules$reach(x$par, ...))
}

#####  Table of rules for handling rmh models ##################

.Spatstat.RmhTable <-
  list(
#
# 0. Poisson (special case)
#
       'poisson'=
       list(
            C.id=NA,
            multitype=FALSE,
            parhandler=function(par, ...) {
              check.named.thing(par, "beta", context="For the Poisson process")
              if(par[["beta"]] < 0)
                stop("Negative value of beta for Poisson process")
              return(par)
            },
            reach = function(par, ...) { return(0) }
            ),
#       
# 1. Strauss.
#       
       'strauss'=
       list(
            C.id="strauss",
            multitype=FALSE,
            parhandler=function(par, ...) {
              par <- check.named.vector(par, c("beta","gamma","r"),
                                        "For the strauss cif")
              if(any(par<0))
		stop("Negative parameters for strauss cif.")
              if(par["gamma"] > 1)
		stop("For Strauss processes, gamma must be <= 1.")
              return(par)
            },
            reach = function(par, ...) {
              r <- par[["r"]]
              g <- par[["gamma"]]
              return(if(g == 1) 0 else r)
            }
            ),
#       
# 2. Strauss with hardcore.
#       
       'straush' =
       list(
            C.id="straush",
            multitype=FALSE,
            parhandler=function(par, ...) {
              par <- check.named.vector(par, c("beta","gamma","r","hc"),
                                        "For the straush cif")
              if(any(par<0))
		stop("Negative parameters for straush cif.")
              return(par)
            },
            reach = function(par, ...) {
              h <- par[["hc"]]
              r <- par[["r"]]
              g <- par[["gamma"]]
              return(if(g == 1) h else r)
            }
            ),
#       
# 3. Softcore.
#
       'sftcr' =
       list(
            C.id="sftcr",
            multitype=FALSE,
            parhandler=function(par, ...) {
              par <- check.named.vector(par, c("beta","sigma","kappa"),
                                        "For the sftcr cif")
              if(any(par<0))
                stop("Negative  parameters for sftcr cif.")
              if(par["kappa"] > 1)
                stop("For Softcore processes, kappa must be <= 1.")
              return(par)
            },
            reach = function(par, ..., epsilon=0) {
              if(epsilon==0)
                return(Inf)
              kappa <- par[["kappa"]]
              sigma <- par[["sigma"]]
              return(sigma/(epsilon^(kappa/2)))
            }
            ),
#       
# 4. Multitype Strauss.
#       
       'straussm' =
       list(
            C.id="straussm",
            multitype=TRUE,
            parhandler=function(par, types) {
              par <- check.named.list(par, c("beta","gamma","radii"),
                                      "For the straussm cif")
              beta <- par$beta
              if(any(is.na(beta)))
		stop("Missing values not allowed in beta.")
              ntypes <- length(types)
              if(length(beta) != ntypes)
		stop("Length of beta does not match ntypes.")
              gamma <- par$gamma
              MultiPair.checkmatrix(gamma, ntypes, "par$gamma")
	      gamma[is.na(gamma)] <- 0

              r <- par$radii
              MultiPair.checkmatrix(r, ntypes, "par$radii")
	      r[is.na(r)] <- 0

# Repack as flat vector
	      par <- c(beta,gamma,r)
              if(any(par<0))
		stop("Some negative parameters for straussm cif.")
              return(par)
            }, 
            reach = function(par, ...) {
              r <- par$radii
              g <- par$gamma
              operative <- ! (is.na(r) | (g == 1))
              return(max(0, r[operative]))
            }
            ),
#       
# 5. Multitype Strauss with hardcore.
#       
       'straushm' = 
       list(
            C.id="straushm",
            multitype=TRUE,
            parhandler=function(par, types) {
              par <- check.named.list(par, c("beta","gamma","iradii","hradii"),
                                      "For the straushm cif")
              beta <- par$beta
              ntypes <- length(types)
              if(length(beta) != ntypes)
		stop("Length of beta does not match ntypes.")
              if(any(is.na(beta)))
		stop("Missing values not allowed in beta.")
              
              gamma <- par$gamma
              MultiPair.checkmatrix(gamma, ntypes, "par$gamma")
              gamma[is.na(gamma)] <- 1

              iradii <- par$iradii
              MultiPair.checkmatrix(iradii, ntypes, "par$iradii")
              iradii[is.na(iradii)] <- 0

              hradii <- par$hradii
              MultiPair.checkmatrix(hradii, ntypes, "par$hradii")
              hradii[is.na(hradii)] <- 0

# Repack as flat vector
              par <- c(beta,gamma,iradii,hradii)
              if(any(par<0))
                  stop("Some negative parameters for straushm cif.")
              return(par)
            },
            reach=function(par, ...) {
              r <- par$iradii
              h <- par$hradii
              g <- par$gamma
              roperative <- ! (is.na(r) | (g == 1))
              hoperative <- ! is.na(h)
              return(max(0, r[roperative], h[hoperative]))
            }
            ),
#       
# 6. Diggle-Gates-Stibbard interaction
#    (function number 1 from Diggle, Gates, and Stibbard)
       
       'dgs' = 
       list(
            C.id="dgs",
            multitype=FALSE,
            parhandler=function(par, ...) {
              par <- check.named.vector(par, c("beta","rho"),
                                        "For the dgs cif")
              if(any(par<0))
		stop("Negative parameters for dgs cif.")
              return(par)
            },
            reach=function(par, ...) {
              return(par[["rho"]])
            }
            ),
#
# 7. Diggle-Gratton interaction 
#    (function number 2 from Diggle, Gates, and Stibbard).

       'diggra' =
       list(
            C.id="diggra",
            multitype=FALSE,
            parhandler=function(par, ...) {
              par <- check.named.vector(par, c("beta","kappa","delta","rho"),
                                      "For the diggra cif")
              if(any(par<0))
		stop("Negative parameters for diggra cif.")
              if(par["delta"] >= par["rho"])
		stop("Radius delta must be less than radius rho.")
              return(par)
            },
            reach=function(par, ...) {
              return(par[["rho"]])
            }
            ),
#       
# 8. Geyer saturation model
#       
       'geyer' = 
       list(
            C.id="geyer",
            multitype=FALSE,
            parhandler=function(par, ...) {
              par <- check.named.vector(par, c("beta","gamma","r","sat"),
                                      "For the geyer cif")
              if(any(par<0))
		stop("Negative parameters for geyer cif.")
              if(par["sat"] > .Machine$integer.max-100)
		par["sat"] <- .Machine$integer.max-100
              return(par)
            },
            reach = function(par, ...) {
              r <- par[["r"]]
              g <- par[["gamma"]]
              return(if(g == 1) 0 else r)
            }
            ),
#       
# 9. The ``lookup'' device.  This permits simulating, at least
# approximately, ANY pairwise interaction function model
# with isotropic pair interaction (i.e. depending only on distance).
# The pair interaction function is provided as a vector of
# distances and corresponding function values which are used
# as a ``lookup table'' by the C code.
#
       'lookup' = 
       list(
            C.id="lookup",
            multitype=FALSE,
            parhandler=function(par, ...) {
              par <- check.named.list(par, c("beta","h"),
                                      "For the lookup cif", "r")
              beta <- par[["beta"]]
              if(beta < 0)
		stop("Negative value of beta for lookup cif.")
              h.init <- par[["h"]]
              r <- par[["r"]]
              if(is.null(r)) {
		if(!is.stepfun(h.init))
                  stop(paste("For cif=lookup, if component r of",
                             "par is absent then component h must",
                             "be a stepfun object."))
		if(!is.cadlag(h.init))
                  stop(paste("The lookup pairwise interaction step",
			     "function must be right continuous,\n",
			     "i.e. built using the default values of the",
                             sQuote("f"), "and", sQuote("right"),
                             "arguments for stepfun."))
		r     <- knots(h.init)
		h0    <- get("yleft",envir=environment(h.init))
		h     <- h.init(r)
		nlook <- length(r)
		if(!identical(all.equal(h[nlook],1),TRUE))
                  stop(paste("The lookup interaction step function",
                             "must be equal to 1 for", dQuote("large"),
                             "distances."))
		if(r[1] <= 0)
                  stop(paste("The first jump point (knot) of the lookup",
                             "interaction step function must be",
                             "strictly positive."))
		h <- c(h0,h)
              } else {
		h     <- h.init
		nlook <- length(r)
		if(length(h) != nlook)
                  stop("Mismatch of lengths of h and r lookup vectors.")
		if(any(is.na(r)))
                  stop("Missing values not allowed in r lookup vector.")
		if(is.unsorted(r))
                  stop("The r lookup vector must be in increasing order.")
		if(r[1] <= 0)
                  stop(paste("The first entry of the lookup vector r",
                             "should be strictly positive."))
		h <- c(h,1)
              }
              if(any(h < 0))
		stop(paste("Negative values in the lookup",
                           "pairwise interaction function."))
              if(h[1] > 0 & any(h > 1))
		stop(paste("Lookup pairwise interaction function does",
                           "not define a valid point process."))
              rmax   <- r[nlook]
              r <- c(0,r)
              nlook <- nlook+1
              deltar <- mean(diff(r))
              if(identical(all.equal(diff(r),rep(deltar,nlook-1)),TRUE)) {
		equisp <- 1
		par <- c(beta,nlook,equisp,deltar,rmax,h)
              } else {
		equisp <- 0
		par <- c(beta,nlook,equisp,deltar,rmax,h,r)
               
              }
              return(par) 
            },
            reach = function(par, ...) {
              r <- par[["r"]]
              h <- par[["h"]]
              if(is.null(r)) 
                r <- knots(h)
              return(max(r))
            }
            ),
#       
# 10. Area interaction
#       
       'areaint'=
       list(
            C.id="areaint",
            multitype=FALSE,
            parhandler=function(par, ...) {
              par <- check.named.vector(par, c("beta","eta","r"),
                                        "For the area interaction cif")
              if(any(par<0))
		stop("Negative parameters for strauss cif.")
              return(par)
            },
            reach = function(par, ...) {
              r <- par[["r"]]
              eta <- par[["eta"]]
              return(if(eta == 1) 0 else (2 * r))
            }
            ),
#
# 11. The ``badgey'' (Baddeley-Geyer) model.
#
       'badgey' =
       list(
            C.id="badgey",
            multitype=FALSE,
            parhandler=function(par, ...) {
              par <- check.named.list(par, c("beta","gamma","r","sat"),
                                      "For the badgey cif")
              beta <- par[["beta"]]
              if(beta < 0)
                stop("Negative value of beta for badgey cif.")
              gamma <- par[["gamma"]]
              r     <- par[["r"]]
              sat   <- par[["sat"]]
              if(length(gamma) != length(r))
                stop(paste("Mismatch between lengths of",
                           dQuote("gamma"), "and", dQuote("r")))
              if(length(sat)==1) sat <- rep(sat,length(gamma))
              else if(length(gamma) != length(sat))
                stop(paste("Mismatch between lengths of",
                           dQuote("gamma"), "and", dQuote("sat")))
              if(any(gamma<0))
                stop(paste("Negative values amongst the",
                           dQuote("gamma"), "parameters"))
              if(any(r<0))
                stop(paste("Negative values amongst the",
                           dQuote("r"), "parameters"))
              if(any(sat<0))
                stop(paste("Negative values amongst the",
                           dQuote("sat"), "parameters"))
              mmm <- cbind(gamma,r,sat)
              mmm <- mmm[order(r),]
              ndisc <- length(r)
              par <- c(par$beta,ndisc,as.vector(t(mmm)))
              return(par)
            },
            reach = function(par, ...) {
              r <- par[["r"]]
              gamma <- par[["gamma"]]
              return(max(r[gamma != 1]))
            }
            )
       # end of list '.Spatstat.RmhTable'
       )

