% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smmparametric.R
\name{smmparametric}
\alias{smmparametric}
\title{Parametric semi-Markov model specification}
\usage{
smmparametric(
  states,
  init,
  ptrans,
  type.sojourn = c("fij", "fi", "fj", "f"),
  distr,
  param,
  cens.beg = FALSE,
  cens.end = FALSE
)
}
\arguments{
\item{states}{Vector of state space of length \eqn{s}.}

\item{init}{Vector of initial distribution of length \eqn{s}.}

\item{ptrans}{Matrix of transition probabilities of the embedded Markov
chain \eqn{J=(J_m)_{m}} of dimension \eqn{(s, s)}.}

\item{type.sojourn}{Type of sojourn time (for more explanations, see Details).}

\item{distr}{\itemize{
\item Matrix of distributions of dimension \eqn{(s, s)} if \code{type.sojourn = "fij"};
\item Vector of distributions of length \eqn{s} if \code{type.sojourn = "fi"} or \verb{"fj};
\item A distribution if \code{type.sojourn = "f"}.
}
where the distributions to be used can be one of \code{unif}, \code{geom}, \code{pois}, \code{dweibull} or \code{nbinom}.}

\item{param}{Parameters of sojourn time distributions:
\itemize{
\item Array of distribution parameters of dimension \eqn{(s, s, 2)}
(2 corresponds to the maximal number of distribution parameters) if \code{type.sojourn = "fij"};
\item Matrix of distribution parameters of dimension \eqn{(s, 2)} if \code{type.sojourn = "fi"} or \code{"fj"};
\item Vector of distribution parameters of length 2 if \code{type.sojourn = "f"}.
}

When parameters/values are not necessary (e.g. the Poisson distribution has
only one parameter that is \eqn{\lambda}, leave the value \code{NA} for the
second parameter in the argument \code{param}).}

\item{cens.beg}{Optional. A logical value indicating whether or not
sequences are censored at the beginning.}

\item{cens.end}{Optional. A logical value indicating whether or not
sequences are censored at the end.}
}
\value{
Returns an object of class \link{smmparametric}.
}
\description{
Creates a parametric model specification for a semi-Markov model.
}
\details{
This function creates a semi-Markov model object in the parametric
case, taking into account the type of sojourn time and the censoring
described in references. For the parametric specification, several discrete
distributions are considered (see below).

The difference between the Markov model and the semi-Markov model concerns
the modeling of the sojourn time. With a Markov chain, the sojourn time
distribution is modeled by a Geometric distribution (in discrete time).
With a semi-Markov chain, the sojourn time can be any arbitrary distribution.
In this package, the available distribution for a semi-Markov model are :
\itemize{
\item Uniform: \eqn{f(x) = 1/n} for \eqn{a \le x \le b}, with \eqn{n = b-a+1};
\item Geometric: \eqn{f(x) = \theta (1-\theta)^x} for \eqn{x = 0, 1, 2,\ldots,n}, \eqn{0 < \theta < 1}, with \eqn{n > 0} and \eqn{\theta} is the probability of success;
\item Poisson: \eqn{f(x) = (\lambda^x exp(-\lambda))/x!} for \eqn{x = 0, 1, 2,\ldots,n}, with \eqn{n > 0} and \eqn{\lambda > 0};
\item Discrete Weibull of type 1: \eqn{f(x)=q^{(x-1)^{\beta}}-q^{x^{\beta}}, x=1,2,3,\ldots,n}, with \eqn{n > 0}, \eqn{q} is the first parameter and \eqn{\beta} is the second parameter;
\item Negative binomial: \eqn{f(x)=\frac{\Gamma(x+\alpha)}{\Gamma(\alpha) x!} p^{\alpha} (1 - p)^x},
for \eqn{x = 0, 1, 2,\ldots,n}, \eqn{\Gamma} is the Gamma function,
\eqn{\alpha} is the parameter of overdispersion and \eqn{p} is the
probability of success, \eqn{0 < p < 1};
\item Non-parametric.
}
We define :
\itemize{
\item the semi-Markov kernel \eqn{q_{ij}(k) = P( J_{m+1} = j, T_{m+1} - T_{m} = k | J_{m} = i )};
\item the transition matrix \eqn{(p_{trans}(i,j))_{i,j} \in states} of the embedded Markov chain \eqn{J = (J_m)_m}, \eqn{p_{trans}(i,j) = P( J_{m+1} = j | J_m = i )};
\item the initial distribution \eqn{\mu_i = P(J_1 = i) = P(Z_1 = i)}, \eqn{i \in 1, 2, \dots, s};
\item the conditional sojourn time distributions \eqn{(f_{ij}(k))_{i,j} \in states,\ k \in N ,\ f_{ij}(k) = P(T_{m+1} - T_m = k | J_m = i, J_{m+1} = j )},
\eqn{f} is specified by the argument \code{param} in the parametric case.
}

In this package we can choose different types of sojourn time.
Four options are available for the sojourn times:
\itemize{
\item depending on the present state and on the next state (\eqn{f_{ij}});
\item depending only on the present state (\eqn{f_{i}});
\item depending only on the next state (\eqn{f_{j}});
\item depending neither on the current, nor on the next state (\eqn{f}).
}

If  \code{type.sojourn = "fij"}, \code{distr} is a matrix of dimension \eqn{(s, s)}
(e.g., if the row 1 of the 2nd column is \code{"pois"}, that is to say we go from
the first state to the second state following a Poisson distribution).
If \code{type.sojourn = "fi"} or \code{"fj"}, \code{distr} must be a vector (e.g., if the
first element of vector is \code{"geom"}, that is to say we go from the first
state to any state according to a Geometric distribution).
If \code{type.sojourn = "f"}, \code{distr} must be one of \code{"unif"}, \code{"geom"}, \code{"pois"},
\code{"dweibull"}, \code{"nbinom"} (e.g., if \code{distr} is equal to \code{"nbinom"}, that is
to say that the sojourn times when going from any state to any state follows
a Negative Binomial distribution).
For the non-parametric case, \code{distr} is equal to \code{"nonparametric"} whatever
type of sojourn time given.

If the sequence is censored at the beginning and/or at the end, \code{cens.beg}
must be equal to \code{TRUE} and/or \code{cens.end} must be equal to \code{TRUE}.
All the sequences must be censored in the same way.
}
\examples{
states <- c("a", "c", "g", "t")
s <- length(states)

# Creation of the initial distribution
vect.init <- c(1 / 4, 1 / 4, 1 / 4, 1 / 4)

# Creation of transition matrix
pij <- matrix(c(0, 0.2, 0.5, 0.3, 
                0.2, 0, 0.3, 0.5, 
                0.3, 0.5, 0, 0.2, 
                0.4, 0.2, 0.4, 0), 
              ncol = s, byrow = TRUE)

# Creation of the distribution matrix

distr.matrix <- matrix(c(NA, "pois", "geom", "nbinom", 
                         "geom", NA, "pois", "dweibull",
                         "pois", "pois", NA, "geom", 
                         "pois", "geom", "geom", NA), 
                       nrow = s, ncol = s, byrow = TRUE)

# Creation of an array containing the parameters
param1.matrix <- matrix(c(NA, 2, 0.4, 4, 
                          0.7, NA, 5, 0.6, 
                          2, 3, NA, 0.6, 
                          4, 0.3, 0.4, NA), 
                        nrow = s, ncol = s, byrow = TRUE)

param2.matrix <- matrix(c(NA, NA, NA, 0.6, 
                          NA, NA, NA, 0.8, 
                          NA, NA, NA, NA, 
                          NA, NA, NA, NA), 
                        nrow = s, ncol = s, byrow = TRUE)

param.array <- array(c(param1.matrix, param2.matrix), c(s, s, 2))

# Specify the semi-Markov model
smm1 <- smmparametric(states = states, init = vect.init, ptrans = pij, 
                      type.sojourn = "fij", distr = distr.matrix, 
                      param = param.array)
smm1

}
\references{
V. S. Barbu, N. Limnios. (2008). Semi-Markov Chains and Hidden Semi-Markov
Models Toward Applications - Their Use in Reliability and DNA Analysis.
New York: Lecture Notes in Statistics, vol. 191, Springer.
}
\seealso{
\link{simulate}, \link{fitsmm}, \link{smmnonparametric}
}
