#' Construct Structural Latent Class Model
#'
#' Constructs a latent structure with multiple latent class variables.
#'
#' @param formula a formula specifying the latent structure. Detailed model specifications are provided under 'Details'.
#' @param ... additional formulae for defining the model structure.
#' @param constraints a list of constraints to enforce measurement invariance. Detailed explanations of applying constraints are available under 'Details'.
#'
#' @details
#' The \strong{`formula`} can be categorized into three types, each serving a distinct purpose:
#'
#' 1. \strong{Defining Latent Class Variables with Manifest Indicators}:
#'    Specify the relationship between a latent class variable and its manifest indicators. The latent class variable is on the left-hand side (lhs), denoted with square brackets \code{[]} or parentheses \code{()} to indicate the number of classes, and manifest indicators are listed on the right-hand side (rhs). For example:
#'    \preformatted{
#' LC1[k] ~ x1 + x2 + x3
#' LC2[k] ~ y1 + y2 + y3
#' LC3(k) ~ z1 + z2 + z3
#'    }
#'    Here, \code{k} denotes the number of latent classes for the variable.

#' 2. \strong{Relating Latent Class Variables to Each Other}:
#'    Define relationships where one latent class variable influences another. For example:
#'    \preformatted{
#' LC2 ~ LC1
#'    }
#'    This formula implies that \code{LC2} is conditionally dependent on \code{LC1}.
#'
#' 3. \strong{Defining Higher-Level Latent Class Variables}:
#'    Specify relationships where a latent class variable is measured by other latent class variables instead of manifest indicators. For example:
#'    \preformatted{
#' P[k] ~ LC1 + LC2 + LC3
#'    }
#'    This indicates that the latent variable \code{P} is measured by the latent class variables \code{LC1}, \code{LC2}, and \code{LC3}.
#'
#' In all formulas, variables on the lhs influence those on the rhs.
#'
#' The \code{constraints} argument enforces specific conditions to ensure precise inference, such as measurement invariance. This is particularly useful for longitudinal analysis (eg. LTA or LCPA), where consistent meanings of latent classes across time are essential.
#'
#' 1. \strong{Measurement Invariance for the Measurement Model}:
#'    Ensures probabilities associated with latent class variables remain consistent. For example:
#'    \preformatted{
#' c("LC1", "LC2", "LC3")
#'    }
#'    This ensures that \code{LC1}, \code{LC2}, and \code{LC3} have semantically consistent measurement probabilities.
#'
#' ' 2. \strong{Measurement Invariance for the Structural Model}:
#'    Applies constraints to ensure consistent interpretations of transition probabilities between latent class variables. For example:
#'    \preformatted{
#' c("P ~ LC1", "P -> LC2")
#'    }
#'    This ensures that the transitions from \code{P} to \code{LC1} and \code{P} to \code{LC2} are consistent.
#'
#' @returns
#' An object of class \code{slca} with the following components:
#' \item{tree}{A \code{data.frame} describing the parent-child relationships among latent class and manifest variables.}
#' \item{latent}{A \code{data.frame} listing all latent class variables with details such as the number of classes.}
#' \item{measure}{A \code{data.frame} describing the measurement model.}
#' \item{struct}{A \code{data.frame} detailing the structural model.}
#'
#' The printed model description is divided into four parts:
#' \enumerate{
#'   \item \strong{Latent variables}: Lists the latent class variables and the number of classes for each variable. The root variable is marked with an asterisk (\code{*}).
#'   \item \strong{Measurement model}: Displays manifest indicators for each latent class variable and any applied measurement constraints (lowercase letters indicate consistency).
#'   \item \strong{Structural model}: Describes the conditional relationships between latent class variables.
#'   \item \strong{Dependency constraints}: Outlines constraints applied to conditional dependencies, where uppercase letters represent consistent dependency structures.
#' }
#'
#'
#' @example man/examples/slca.R
#'
#' @export
slca <- function(
   formula = NULL, ..., constraints = NULL
) {
   if (!is.list(formula)) formulae <- list(formula, ...)
   else formulae <- formula
   formulae <- formulae[sapply(formulae, inherits, "formula")]
   model <- proc_formula(formulae, constraints)
   arg <- arguments(model)

   res <- list()
   res$model <- model
   res$arg <- arg

   class(res) <- "slca"
   res
}
