% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/variable_selection.R
\name{VariableSelection}
\alias{VariableSelection}
\title{Stability selection in regression}
\usage{
VariableSelection(
  xdata,
  ydata = NULL,
  Lambda = NULL,
  pi_list = seq(0.6, 0.9, by = 0.01),
  K = 100,
  tau = 0.5,
  seed = 1,
  n_cat = 3,
  family = "gaussian",
  implementation = PenalisedRegression,
  resampling = "subsampling",
  cpss = FALSE,
  PFER_method = "MB",
  PFER_thr = Inf,
  FDP_thr = Inf,
  Lambda_cardinal = 100,
  group_x = NULL,
  group_penalisation = FALSE,
  n_cores = 1,
  output_data = FALSE,
  verbose = TRUE,
  beep = NULL,
  ...
)
}
\arguments{
\item{xdata}{matrix of predictors with observations as rows and variables as
columns.}

\item{ydata}{optional vector or matrix of outcome(s). If \code{family} is set
to \code{"binomial"} or \code{"multinomial"}, \code{ydata} can be a vector
with character/numeric values or a factor.}

\item{Lambda}{matrix of parameters controlling the level of sparsity in the
underlying feature selection algorithm specified in \code{implementation}.
If \code{Lambda=NULL} and \code{implementation=PenalisedRegression},
\code{\link{LambdaGridRegression}} is used to define a relevant grid.}

\item{pi_list}{vector of thresholds in selection proportions. If
\code{n_cat=3}, these values must be \code{>0.5} and \code{<1}. If
\code{n_cat=2}, these values must be \code{>0} and \code{<1}.}

\item{K}{number of resampling iterations.}

\item{tau}{subsample size. Only used if \code{resampling="subsampling"} and
\code{cpss=FALSE}.}

\item{seed}{value of the seed to initialise the random number generator and
ensure reproducibility of the results (see \code{\link[base]{set.seed}}).}

\item{n_cat}{number of categories used to compute the stability score.
Possible values are 2 or 3.}

\item{family}{type of regression model. This argument is defined as in
\code{\link[glmnet]{glmnet}}. Possible values include \code{"gaussian"}
(linear regression), \code{"binomial"} (logistic regression),
\code{"multinomial"} (multinomial regression), and \code{"cox"} (survival
analysis).}

\item{implementation}{function to use for variable selection. Possible
functions are: \code{PenalisedRegression}, \code{SparsePLS},
\code{GroupPLS} and \code{SparseGroupPLS}. Alternatively, a user-defined
function can be provided.}

\item{resampling}{resampling approach. Possible values are:
\code{"subsampling"} for sampling without replacement of a proportion
\code{tau} of the observations, or \code{"bootstrap"} for sampling with
replacement generating a resampled dataset with as many observations as in
the full sample. Alternatively, this argument can be a function to use for
resampling. This function must use arguments named \code{data} and
\code{tau} and return the IDs of observations to be included in the
resampled dataset.}

\item{cpss}{logical indicating if complementary pair stability selection
should be done. For this, the algorithm is applied on two non-overlapping
subsets of half of the observations. A feature is considered as selected if
it is selected for both subsamples. With this method, the data is split
\code{K/2} times (\code{K} models are fitted). Only used if
\code{PFER_method="MB"}.}

\item{PFER_method}{method used to compute the upper-bound of the expected
number of False Positives (or Per Family Error Rate, PFER). If
\code{PFER_method="MB"}, the method proposed by Meinshausen and Bühlmann
(2010) is used. If \code{PFER_method="SS"}, the method proposed by Shah and
Samworth (2013) under the assumption of unimodality is used.}

\item{PFER_thr}{threshold in PFER for constrained calibration by error
control. If \code{PFER_thr=Inf} and \code{FDP_thr=Inf}, unconstrained
calibration is used (the default).}

\item{FDP_thr}{threshold in the expected proportion of falsely selected
features (or False Discovery Proportion) for constrained calibration by
error control. If \code{PFER_thr=Inf} and \code{FDP_thr=Inf}, unconstrained
calibration is used (the default).}

\item{Lambda_cardinal}{number of values in the grid of parameters controlling
the level of sparsity in the underlying algorithm. Only used if
\code{Lambda=NULL}.}

\item{group_x}{vector encoding the grouping structure among predictors. This
argument indicates the number of variables in each group. Only used for
models with group penalisation (e.g. \code{implementation=GroupPLS} or
\code{implementation=SparseGroupPLS}).}

\item{group_penalisation}{logical indicating if a group penalisation should
be considered in the stability score. The use of
\code{group_penalisation=TRUE} strictly applies to group (not sparse-group)
penalisation.}

\item{n_cores}{number of cores to use for parallel computing (see
\code{\link[parallel]{mclapply}}). Only available on Unix systems.}

\item{output_data}{logical indicating if the input datasets \code{xdata} and
\code{ydata} should be included in the output.}

\item{verbose}{logical indicating if a loading bar and messages should be
printed.}

\item{beep}{sound indicating the end of the run. Possible values are:
\code{NULL} (no sound) or an integer between 1 and 11 (see argument
\code{sound} in \code{\link[beepr]{beep}}).}

\item{...}{additional parameters passed to the functions provided in
\code{implementation} or \code{resampling}.}
}
\value{
An object of class \code{variable_selection}. A list with: \item{S}{a
  matrix of the best stability scores for different parameters controlling
  the level of sparsity in the underlying algorithm.} \item{Lambda}{a matrix
  of parameters controlling the level of sparsity in the underlying
  algorithm.} \item{Q}{a matrix of the average number of selected features by
  the underlying algorithm with different parameters controlling the level of
  sparsity.} \item{Q_s}{a matrix of the calibrated number of stably selected
  features with different parameters controlling the level of sparsity.}
  \item{P}{a matrix of calibrated thresholds in selection proportions for
  different parameters controlling the level of sparsity in the underlying
  algorithm.} \item{PFER}{a matrix of upper-bounds in PFER of calibrated
  stability selection models with different parameters controlling the level
  of sparsity.} \item{FDP}{a matrix of upper-bounds in FDP of calibrated
  stability selection models with different parameters controlling the level
  of sparsity.} \item{S_2d}{a matrix of stability scores obtained with
  different combinations of parameters. Columns correspond to different
  thresholds in selection proportions.} \item{PFER_2d}{a matrix of
  upper-bounds in FDP obtained with different combinations of parameters.
  Columns correspond to different thresholds in selection proportions.}
  \item{FDP_2d}{a matrix of upper-bounds in PFER obtained with different
  combinations of parameters. Columns correspond to different thresholds in
  selection proportions.} \item{selprop}{a matrix of selection proportions.
  Columns correspond to predictors from \code{xdata}.} \item{Beta}{an array
  of model coefficients. Columns correspond to predictors from \code{xdata}.
  Indices along the third dimension correspond to different resampling
  iterations. With multivariate outcomes, indices along the fourth dimension
  correspond to outcome-specific coefficients.} \item{method}{a list with
  \code{type="variable_selection"} and values used for arguments
  \code{implementation}, \code{family}, \code{resampling}, \code{cpss} and
  \code{PFER_method}.} \item{params}{a list with values used for arguments
  \code{K}, \code{pi_list}, \code{tau}, \code{n_cat}, \code{pk}, \code{n}
  (number of observations), \code{PFER_thr}, \code{FDP_thr} and \code{seed}.
  The datasets \code{xdata} and \code{ydata} are also included if
  \code{output_data=TRUE}.} For all matrices and arrays returned, the rows
  are ordered in the same way and correspond to parameter values stored in
  \code{Lambda}.
}
\description{
Performs stability selection for regression models. The underlying variable
selection algorithm (e.g. LASSO regression) is run with different
combinations of parameters controlling the sparsity (e.g. penalty parameter)
and thresholds in selection proportions. These two hyper-parameters are
jointly calibrated by maximisation of the stability score.
}
\details{
In stability selection, a feature selection algorithm is fitted on
  \code{K} subsamples (or bootstrap samples) of the data with different
  parameters controlling the sparsity (\code{Lambda}). For a given (set of)
  sparsity parameter(s), the proportion out of the \code{K} models in which
  each feature is selected is calculated. Features with selection proportions
  above a threshold pi are considered stably selected. The stability
  selection model is controlled by the sparsity parameter(s) for the
  underlying algorithm, and the threshold in selection proportion:

  \eqn{V_{\lambda, \pi} = \{ j: p_{\lambda}(j) \ge \pi \} }

  If argument \code{group_penalisation=FALSE}, "feature" refers to variable
  (variable selection model). If argument \code{group_penalisation=TRUE},
  "feature" refers to group (group selection model). In this case, groups
  need to be defined \emph{a priori} and specified in argument
  \code{group_x}.

  These parameters can be calibrated by maximisation of a stability score
  (see \code{\link{StabilityScore}}) derived from the likelihood under the
  assumption of uniform (uninformative) selection:

  \eqn{S_{\lambda, \pi} = -log(L_{\lambda, \pi})}

  It is strongly recommended to examine the calibration plot carefully to
  check that the grids of parameters \code{Lambda} and \code{pi_list} do not
  restrict the calibration to a region that would not include the global
  maximum (see \code{\link{CalibrationPlot}}). In particular, the grid
  \code{Lambda} may need to be extended when the maximum stability is
  observed on the left or right edges of the calibration heatmap.

  To control the expected number of False Positives (Per Family Error Rate)
  in the results, a threshold \code{PFER_thr} can be specified. The
  optimisation problem is then constrained to sets of parameters that
  generate models with an upper-bound in PFER below \code{PFER_thr} (see
  Meinshausen and Bühlmann (2010) and Shah and Samworth (2013)).

  Possible resampling procedures include defining (i) \code{K} subsamples of
  a proportion \code{tau} of the observations, (ii) \code{K} bootstrap
  samples with the full sample size (obtained with replacement), and (iii)
  \code{K/2} splits of the data in half for complementary pair stability
  selection (see arguments \code{resampling} and \code{cpss}). In
  complementary pair stability selection, a feature is considered selected at
  a given resampling iteration if it is selected in the two complementary
  subsamples.

  For categorical or time to event outcomes (argument \code{family} is
  \code{"binomial"}, \code{"multinomial"} or \code{"cox"}), the proportions
  of observations from each category in all subsamples or bootstrap samples
  are the same as in the full sample.

  To ensure reproducibility of the results, the starting number of the random
  number generator is set to \code{seed}.

  For parallelisation, stability selection with different sets of parameters
  can be run on \code{n_cores} cores. This relies on forking with
  \code{\link[parallel]{mclapply}} (specific to Unix systems). Alternatively,
  the function can be run manually with different \code{seed}s and all other
  parameters equal. The results can then be combined using
  \code{\link{Combine}}.
}
\examples{
\donttest{
oldpar <- par(no.readonly = TRUE)
par(mar = rep(7, 4))

# Linear regression
set.seed(1)
simul <- SimulateRegression(n = 100, pk = 50, family = "gaussian")
stab <- VariableSelection(xdata = simul$xdata, ydata = simul$ydata, family = "gaussian")
print(stab)
CalibrationPlot(stab)
summary(stab)
SelectedVariables(stab)
plot(stab)

# Using additional arguments from glmnet (e.g. penalty.factor)
stab <- VariableSelection(
  xdata = simul$xdata, ydata = simul$ydata, family = "gaussian",
  penalty.factor = c(rep(1, 45), rep(0, 5))
)
summary(stab)

# Regression with multivariate outcomes
set.seed(1)
simul <- SimulateRegression(n = 100, pk = 20, q = 3, family = "gaussian")
stab <- VariableSelection(xdata = simul$xdata, ydata = simul$ydata, family = "mgaussian")
summary(stab)

# Logistic regression
set.seed(1)
simul <- SimulateRegression(n = 200, pk = 10, family = "binomial", ev_xy = 0.8)
stab <- VariableSelection(xdata = simul$xdata, ydata = simul$ydata, family = "binomial")
summary(stab)

# Sparse PCA (1 component, see BiSelection for more components)
set.seed(1)
simul <- SimulateComponents(pk = c(5, 3, 4))
stab <- VariableSelection(
  xdata = simul$data,
  Lambda = 1:(ncol(simul$data) - 1),
  implementation = SparsePCA
)
CalibrationPlot(stab, xlab = "")
summary(stab)

# Sparse PLS (1 outcome, 1 component, see BiSelection for more options)
set.seed(1)
simul <- SimulateRegression(n = 100, pk = 50, family = "gaussian")
stab <- VariableSelection(
  xdata = simul$xdata, ydata = simul$ydata,
  Lambda = 1:(ncol(simul$xdata) - 1),
  implementation = SparsePLS, family = "gaussian"
)
CalibrationPlot(stab, xlab = "")
SelectedVariables(stab)

# Group PLS (1 outcome, 1 component, see BiSelection for more options)
stab <- VariableSelection(
  xdata = simul$xdata, ydata = simul$ydata,
  Lambda = 1:5,
  group_x = c(5, 5, 10, 20, 10),
  group_penalisation = TRUE,
  implementation = GroupPLS, family = "gaussian"
)
CalibrationPlot(stab, xlab = "")
SelectedVariables(stab)

# Sparse PLS-DA (1 outcome, 1 component, see BiSelection for more options)
set.seed(1)
simul <- SimulateRegression(n = 200, pk = 20, family = "binomial")
stab <- VariableSelection(
  xdata = simul$xdata, ydata = simul$ydata,
  Lambda = 1:(ncol(simul$xdata) - 1),
  implementation = SparsePLS,
  family = "binomial"
)
CalibrationPlot(stab, xlab = "")
summary(stab)

# Example with more hyper-parameters: elastic net
set.seed(1)
simul <- SimulateRegression(n = 100, pk = 50, family = "gaussian")
TuneElasticNet <- function(xdata, ydata, family, alpha) {
  stab <- VariableSelection(
    xdata = xdata, ydata = ydata,
    family = family, alpha = alpha, verbose = FALSE
  )
  return(max(stab$S, na.rm = TRUE))
}
myopt <- optimise(TuneElasticNet,
  lower = 0.1, upper = 1, maximum = TRUE,
  xdata = simul$xdata, ydata = simul$ydata,
  family = "gaussian"
)
stab <- VariableSelection(
  xdata = simul$xdata, ydata = simul$ydata,
  family = "gaussian", alpha = myopt$maximum
)
summary(stab)
enet <- SelectedVariables(stab)

# Comparison with LASSO
stab <- VariableSelection(xdata = simul$xdata, ydata = simul$ydata, family = "gaussian")
summary(stab)
lasso <- SelectedVariables(stab)
table(lasso, enet)

# Example using an external function: group-LASSO with gglasso
if (requireNamespace("gglasso", quietly = TRUE)) {
  set.seed(1)
  simul <- SimulateRegression(n = 200, pk = 20, family = "binomial")
  ManualGridGroupLasso <- function(xdata, ydata, family, group_x, ...) {
    # Defining the grouping
    group <- do.call(c, lapply(1:length(group_x), FUN = function(i) {
      rep(i, group_x[i])
    }))

    if (family == "binomial") {
      ytmp <- ydata
      ytmp[ytmp == min(ytmp)] <- -1
      ytmp[ytmp == max(ytmp)] <- 1
      return(gglasso::gglasso(xdata, ytmp, loss = "logit", group = group, ...))
    } else {
      return(gglasso::gglasso(xdata, ydata, lambda = lambda, loss = "ls", group = group, ...))
    }
  }
  Lambda <- LambdaGridRegression(
    xdata = simul$xdata, ydata = simul$ydata,
    family = "binomial", Lambda_cardinal = 20,
    implementation = ManualGridGroupLasso,
    group_x = rep(5, 4)
  )
  GroupLasso <- function(xdata, ydata, Lambda, family, group_x, ...) {
    # Defining the grouping
    group <- do.call(c, lapply(1:length(group_x), FUN = function(i) {
      rep(i, group_x[i])
    }))

    # Running the regression
    if (family == "binomial") {
      ytmp <- ydata
      ytmp[ytmp == min(ytmp)] <- -1
      ytmp[ytmp == max(ytmp)] <- 1
      mymodel <- gglasso::gglasso(xdata, ytmp, lambda = Lambda, loss = "logit", group = group, ...)
    }
    if (family == "gaussian") {
      mymodel <- gglasso::gglasso(xdata, ydata, lambda = Lambda, loss = "ls", group = group, ...)
    }
    # Extracting and formatting the beta coefficients
    beta_full <- t(as.matrix(mymodel$beta))
    beta_full <- beta_full[, colnames(xdata)]

    selected <- ifelse(beta_full != 0, yes = 1, no = 0)

    return(list(selected = selected, beta_full = beta_full))
  }
  stab <- VariableSelection(
    xdata = simul$xdata, ydata = simul$ydata,
    implementation = GroupLasso, family = "binomial", Lambda = Lambda,
    group_x = rep(5, 4),
    group_penalisation = TRUE
  )
  summary(stab)
}

par(oldpar)
}

}
\references{
\insertRef{ourstabilityselection}{sharp}

  \insertRef{stabilityselectionSS}{sharp}

  \insertRef{stabilityselectionMB}{sharp}

  \insertRef{lasso}{sharp}
}
\seealso{
\code{\link{PenalisedRegression}}, \code{\link{SelectionAlgo}},
  \code{\link{LambdaGridRegression}}, \code{\link{Resample}},
  \code{\link{StabilityScore}} \code{\link{Refit}},
  \code{\link{ExplanatoryPerformance}}, \code{\link{Incremental}},

Other stability functions: 
\code{\link{BiSelection}()},
\code{\link{Clustering}()},
\code{\link{GraphicalModel}()}
}
\concept{stability functions}
