% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/influ_phyglm.R
\name{influ_phyglm}
\alias{influ_phyglm}
\title{Influential species detection - Phylogenetic Logistic Regression}
\usage{
influ_phyglm(formula, data, phy, btol = 50, cutoff = 2, track = TRUE, ...)
}
\arguments{
\item{formula}{The model formula}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{btol}{Bound on searching space. For details see \code{phyloglm}}

\item{cutoff}{The cutoff value used to identify for influential species
(see Details)}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{phyloglm}}
}
\value{
The function \code{influ_phyglm} returns a list with the following
components:

\code{cutoff}: The value selected for \code{cutoff}

\code{formula}: The formula

\code{full.model.estimates}: Coefficients, aic and the optimised
value of the phylogenetic parameter (i.e. \code{alpha}) for the full model
without deleted species.

\code{influential_species}: List of influential species, both
based on standardised difference in intercept and in the slope of the
regression. Species are ordered from most influential to less influential and
only include species with a standardised difference > \code{cutoff}.

\code{sensi.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade. Columns report the calculated
regression intercept (\code{intercept}), difference between simulation
intercept and full model intercept (\code{DIFintercept}), the standardised
difference (\code{sDIFintercept}), the percentage of change in intercept compared
to the full model (\code{intercept.perc}) and intercept p-value
(\code{pval.intercept}). All these parameters are also reported for the regression
slope (\code{DIFestimate} etc.). Additionally, model aic value (\code{AIC}) and
the optimised value (\code{optpar}) of the phylogenetic parameter
(i.e. \code{alpha}) are reported.

\code{data}: Original full dataset.

\code{errors}: Species where deletion resulted in errors.
}
\description{
Performs leave-one-out deletion analysis for phylogenetic logistic regression,
and detects influential species.
}
\details{
This function sequentially removes one species at a time, fits a phylogenetic
logistic regression model using \code{\link[phylolm]{phyloglm}}, stores the
results and detects influential species.

Currently only logistic regression using the "logistic_MPLE"-method from
\code{phyloglm} is implemented.

\code{influ_phyglm} detects influential species based on the standardised
difference in intercept and/or slope when removing a given species compared
to the full model including all species. Species with a standardised difference
above the value of \code{cutoff} are identified as influential. The default
value for the cutoff is 2 standardised differences change.

Currently, this function can only implement simple logistic models (i.e. \eqn{trait~
predictor}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\examples{
# Simulate Data:
set.seed(6987)
phy = rtree(100)
x = rTrait(n=1,phy=phy)
X = cbind(rep(1,100),x)
y = rbinTrait(n=1,phy=phy, beta=c(-1,0.5), alpha=.7 ,X=X)
dat = data.frame(y, x)
# Run sensitivity analysis:
influ <- influ_phyglm(y ~ x, data = dat, phy = phy) 
# To check summary results and most influential species:
summary(influ)
# Visual diagnostics for clade removal:
sensi_plot(influ)
}
\references{
Paterno, G. B., Penone, C. Werner, G. D. A. 
\href{http://doi.wiley.com/10.1111/2041-210X.12990}{sensiPhy: 
An r-package for sensitivity analysis in phylogenetic 
comparative methods.} Methods in Ecology and Evolution 
2018, 9(6):1461-1467.  

Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phyloglm}}, \code{\link{samp_phyglm}},
\code{\link{influ_phylm}}, \code{\link{sensi_plot}}
}
\author{
Gustavo Paterno & Gijsbert D.A. Werner
}
