% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelcomparison.R
\name{compareLavaan}
\alias{compareLavaan}
\title{Prepare a table to compare fit measures of confirmatory factor analyses (CFA)}
\usage{
compareLavaan(models, fitmeas = c("chisq", "df", "pvalue", "rmsea",
  "cfi", "tli", "srmr", "aic", "bic"), nesting = NULL, scaled = TRUE,
  chidif = TRUE, digits = 3, ...)
}
\arguments{
\item{models}{list of lavaan cfa or sem models. Model names can be
supplied. See examples.}

\item{fitmeas}{A vector of fit measures. One or more of these
\code{c("chisq", "df", "pvalue", "rmsea", "cfi", "tli",
"srmr", "aic", "bic", "srmr", "aic", "bic",
"srmr_mplus")}. Other fit measures present in the lavaan
objects will be allowed; fit measures that are requested but
not found are ignored.}

\item{nesting}{character string indicating the nesting structure
of the models.  Must only contain model names, ">", and "+"
separated by spaces.  The model to the left of a ">" is the
parent model for all models to the right of the same ">", up
until another ">" is reached. When multiple models are nested
in the same parent, they are separated by a "+".}

\item{scaled}{should scaled versions of the fit measures requested
be used if available?  The scaled statistic is determined by
the model estimation method.  The defaul value is TRUE.}

\item{chidif}{should the nested models be compared by using the
anova function? The anova function may pass the model
comparison on to another lavaan function.  The results are
added to the last three columns of the comparison table. The
default value is TRUE.}

\item{digits}{The digits argument that will be passed to xtable.}

\item{...}{Arguments that will be passed to print.xtable. These
arguments can be used to control table caption, label, and so
forth. See \code{?print.xtable}.  If \code{type = "latex"} or
\code{"html"}, this function sets additional default values
for print.xtable that can be overridden by specifying
arguments here. Default type is an R data.frame, which is printed on
screen. Note the print.xtable parameter \code{print.results} determines
whether the markup is displayed before it is returned. The \code{file}
parameter can specify a file into which the markup is to be saved.}
}
\value{
If type = NULL, a data.frame object includes an attribute
    called "noteinfo". If type = "tex", return is a character
    vector created by xtable. If type = "html", a vector of HTML
    markup created by xtable.
}
\description{
If the parameter \code{nesting} is not specified, then this
function attempts to discern which models are nested and they are
ordered accordingly. The user can override that by specifing a
nesting structure. This uses a new notation to represent nesting
of models.  See the parameter \code{nesting}.  The syntax uses the
symbols ">" and "+" in an obvious way to indicate that one model
is the superset or on the same level as another. If the
}
\details{
In May 2018, the output approach was changed. The functions
\code{xtable} and \code{print.xtable} are used to render the final
result and any of the arguments allowed by \code{print.xtable} can
be used here (via the \code{...} argument). We have some default
settings for some \code{print.xtable}, such as \code{type = NULL},
\code{file = NULL}, \code{print.results = TRUE}, and
\code{math.style.exponents = TRUE}. There are some other specific
defaults for type = "latex" and type = "html", but they can all be
overridden by the user. We include a model legend at the bottom of
the table, indicating which models are compared by the Chi-squared
statistic.

If the type argument is not specified, then the output will be a
simple text display of the model table.  If type is either "latex"
or "html", then a marked up table will be displayed and the file
argument can be used to ask for a saved version. If the user wants
to simply save the result in a file, and not display on screen,
insert the argument \code{print.results = FALSE}.
}
\examples{
library(lavaan)
library(xtable)
set.seed(123)
genmodel <- "f1 =~ .7*v1 + .7*v2 + .7*v3 + .7*v4 + .7*v5 + .7*v6
f1 ~~ 1*f1"
genmodel2 <- "f1 =~ .7*v1 + .7*v2 + .7*v3 + .7*v4 + .7*v5 + .2*v6
f1 ~~ 1*f1"
##'
dat1 <- simulateData(genmodel, sample.nobs = 300)
dat2 <- simulateData(genmodel2, sample.nobs = 300)
dat1$group <- 0
dat2$group <- 1
dat <- rbind(dat1, dat2)
## In order from less constrained to restricted
## Model cc1 configural model
cc1.model <- "
              f1 =~ 1*v1 + v2 + v3 + v4 + v5 + v6
    		  f1 ~~ f1
    		  f1 ~0*1
    		 "
## Model2: cc2 partial weak model (AKA partial metric)
cc2.model <- "
              f1 =~ 1*v1 + c(L2,L2)*v2 + c(L3,L3)*v3 + c(L4,L4)*v4 + c(L5,L5)*v5 + v6
    		  f1 ~~ f1
    		  f1 ~0*1
    		"
## Model 3: weak model (AKA metric)
cc3.model <- "
            f1 =~ 1*v1 + c(L2,L2)*v2 + c(L3,L3)*v3 + c(L4,L4)*v4 + c(L5,L5)*v5 + c(L6,L6)*v6
    		f1 ~~ f1
    		 f1 ~0*1
    		"
## Model 4: scalar model (AKA strong)
cc4.model <- "
              f1 =~ 1*v1 + c(L2,L2)*v2 + c(L3,L3)*v3 + c(L4,L4)*v4 + c(L5,L5)*v5 + v6
    		  f1 ~~ f1
    		  f1 ~ c(0,NA)*1
    		  v1 ~ c(I1,I1)*1
    		  v2 ~ c(I2,I2)*1
    		  v3 ~ c(I3,I3)*1
    		  v4 ~ c(I4,I4)*1
    		  v5 ~ c(I5,I5)*1
    		  v6 ~ c(I6,I6)*1
    		"

## Reload saved models if available: avoids slow re-estimation that bothers CRAN
cc1 <-  tryCatch(readRDS(system.file("cfa/cc1.rds", package = "semTable")),
                 error = function(e) cfa(cc1.model, data=dat, group="group",
                                         meanstructure=TRUE, estimator = "MLR"))
cc2 <- tryCatch(readRDS(system.file("cfa/cc2.rds", package = "semTable")),
                error = function(e) cfa(cc2.model, data=dat, group="group",
                                        meanstructure=TRUE, estimator = "MLR"))
cc3 <- tryCatch(readRDS(system.file("cfa/cc3.rds", package = "semTable")),
                 error = function(e) cfa(cc3.model, data=dat, group="group",
                                         meanstructure=TRUE, estimator = "MLR"))
cc4 <- tryCatch(readRDS(system.file("cfa/cc4.rds", package = "semTable")),
                error = function(e) cfa(cc4.model, data=dat, group="group",
                                        meanstructure=TRUE, estimator = "MLR"))

models <- list(cc1, cc2, cc3, cc4)
## Note, nesting is not specified, so built-in nesting detection applies
compareLavaan(models)
compareLavaan(models, type = "latex")
compareLavaan(models, type = "html")
##'
## Now we specify model labels in the list
models <- list("Configural" = cc1,  "PartialMetric" = cc2, "Metric" = cc3, "Scalar" = cc4)
## The model labels are used in the nesting parameter
compareLavaan(models, nesting = "Configural > PartialMetric > Metric > Scalar")
##' Previous incorrect, treat cc2 and cc3 as children of cc1 instead:
compareLavaan(models, nesting = "Configural > PartialMetric + Metric > Scalar")
##' 
compareLavaan(models, fitmeas = c("chisq", "df", "cfi", "rmsea", "tli"),
              nesting = "Configural > Metric + PartialMetric > Scalar")

compareLavaan(models, fitmeas = c("chisq", "df", "cfi", "rmsea", "tli"),
             nesting = "Configural > PartialMetric + Metric > Scalar")
##'
## Creates output file
## compareLavaan(models, fitmeas = c("chisq", "df", "cfi", "rmsea", "tli"),
## nesting = "Configural > Metric + PartialMetric > Scalar", type = "tex",
## file = "/tmp/table.tex")
}
\author{
Ben Kite <bakite@ku.edu> and Paul Johnson
    <pauljohn@ku.edu>
}
