% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/function-map-total-n.R
\name{function_map_total_n}
\alias{function_map_total_n}
\title{Create new \verb{*_map_total_n()} functions}
\usage{
function_map_total_n(
  .fun,
  .reported,
  .name_test,
  .name_class = NULL,
  .dispersion = 0:5,
  .n_min = 1L,
  .n_max = NULL,
  .constant = NULL,
  .constant_index = NULL
)
}
\arguments{
\item{.fun}{Function such as \code{grim_map}: It will be used to test columns in a
data frame for consistency. Test results are Boolean and need to be
contained in a column called \code{consistency} that is added to the input data
frame. This modified data frame is then returned by \code{.fun}.}

\item{.reported}{String. Names of the columns containing group-specific
statistics that were reported alongside the total sample size(s). They will
be tested for consistency with the hypothetical group sizes. Examples are
\code{"x"} for GRIM and \code{c("x", "sd")} for DEBIT. In the data frame with
reported group statistics that the manufactured function takes as an input,
each will need to fan out like \code{"x1"}, \code{"x2"}, \code{"sd1"}, and \code{"sd2"}.}

\item{.name_test}{String (length 1). The name of the consistency test, such
as \code{"GRIM"}, to be optionally shown in a message when using the
manufactured function.}

\item{.name_class}{String. If specified, the tibbles returned by the
manufactured function will inherit this string as an S3 class. Default is
\code{NULL}, i.e., no extra class.}

\item{.dispersion, .n_min, .n_max, .constant, .constant_index}{Arguments passed
down to \code{disperse_total()}, using defaults from there.}
}
\value{
A function such as these:\tabular{lll}{
   \strong{Manufactured function} \tab \strong{Reported statistics} \tab \strong{Test vignette} \cr
   \code{grim_map_total_n()} \tab \code{"x"} \tab \code{vignette("grim")} \cr
   \code{grimmer_map_total_n()} \tab \code{"x"}, \code{"sd"} \tab \code{vignette("grimmer")} \cr
   \code{debit_map_total_n()} \tab \code{"x"}, \code{"sd"} \tab \code{vignette("debit")} \cr
}


The factory-made function will also have dots, \code{...}, to pass arguments
down to \code{.fun}, i.e., the basic mapper function.
}
\description{
\code{function_map_total_n()} is the engine that powers functions
such as \code{grim_map_total_n()}. It creates new, "manufactured" functions for
consistency tests. The new functions take reported summary statistics such
as means and apply those tests in cases where only a total sample size is
known, not group sizes.

This works by making \code{disperse_total()} create multiple pairs of
hypothetical group sizes, all of which add up to the reported total. There
need to be exactly two groups.

If functions created this way are exported from other packages, they should
be written as if they were created with
\href{https://purrr.tidyverse.org/reference/faq-adverbs-export.html}{purrr
adverbs}; see explanations there and examples at
\code{vignette("consistency-tests")}, section \emph{Creating mappers with
\code{function_map()}}.
}
\details{
This function is a so-called function factory: It produces other
functions, such as \code{grim_map_total_n()}. More specifically, it is a
function operator (a.k.a. decorator) because it also takes functions as
inputs, such as \code{grim_map()}. See Wickham (2019, ch. 10-11).
}
\section{Conventions}{
 The name of a function manufactured with
\code{function_map_total_n()} should mechanically follow from that of the input
function. For example, \code{grim_map_total_n()} derives from \code{grim_map()}.
This pattern fits best if the input function itself is named after the test
it performs on a data frame, followed by \verb{_map}: \code{grim_map()} applies GRIM,
\code{debit_map()} applies DEBIT, etc.

Much the same is true for the classes of data frames returned by the
manufactured function via the \code{.name_class} argument of
\code{function_map_total_n()}. It should be the function's own name preceded by
the name of the package that contains it or by an acronym of that package's
name. In this way, existing classes are \code{scr_grim_map_total_n} and
\code{scr_debit_map_total_n}.
}

\examples{
# Function definition of `grim_map_total_n()`:
grim_map_total_n <- function_map_total_n(
  .fun = grim_map,
  .reported = "x",
  .name_test = "GRIM",
  .name_class = "scr_grim_map_total_n"
)


# Case study of SCHLIM, a new consistency test --------------

# (Note: This is a mock test without any real significance.
# Its only purpose is to show the minimal steps necessary
# for implementing a serious consistency test, and to use
# it as a starting point for `function_map_total_n()`.)

# The "SCHLIM test" is analogous to GRIM as implemented
# in scrutiny. This is also true for the function names.
# Note that the analogue to `schlim_scalar()`, a function
# called `grim_scalar()`, is not exported from scrutiny,
# but used internally for `grim()`, `grim_map()`, and,
# indirectly, `grim_map_total_n()`.

# Basic test implementation:
schlim_scalar <- function(y, n) {
  (y / 3) > n
}

# This step is not needed below, but
# included for completeness:
schlim <- Vectorize(schlim_scalar)

# This will be the input function for
# `function_map_total_n()`:
schlim_map <- function_map(
  .fun = schlim_scalar,
  .reported = c("y", "n"),
  .name_test = "SCHLIM"
)

# Fire up the function factory:
schlim_map_total_n <- function_map_total_n(
  .fun = schlim_map,
  .reported = "y",
  .name_test = "SCHLIM",
)

# Create some example data:
df1 <- tibble::tibble(
  y1 = 16:25,
  y2 = 26:35,
  n  = 12:21
)
df1

# Call the manufactured function:
out <- schlim_map_total_n(df1)
out

# Summarize the results:
audit_total_n(out)
}
\references{
Bauer, P. J., & Francis, G. (2021). Expression of Concern: Is It
Light or Dark? Recalling Moral Behavior Changes Perception of Brightness.
\emph{Psychological Science}, 32(12), 2042–2043.
https://journals.sagepub.com/doi/10.1177/09567976211058727

Wickham, H. (2019). \emph{Advanced R} (Second Edition). CRC Press/Taylor and
Francis Group. https://adv-r.hadley.nz/index.html
}
\seealso{
\code{disperse_total()}
}
