% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rmargint-fn.R
\name{margint.cl}
\alias{margint.cl}
\title{Classic marginal integration procedures for additive models}
\usage{
margint.cl(Xp, yp, point = NULL, windows, epsilon = 1e-06,
  prob = NULL, type = "0", degree = NULL, qderivate = FALSE,
  orderkernel = 2, Qmeasure = NULL)
}
\arguments{
\item{Xp}{Matrix (n by p) of explanatory variables.}

\item{yp}{Vector of responses (missing values are allowed).}

\item{point}{Matrix of points where predictions will be computed and returned.}

\item{windows}{Vector or a squared matrix of bandwidths for the smoothing estimation procedure.}

\item{epsilon}{Convergence criterion.}

\item{prob}{Vector of robabilities of observing each response (n). Defaults to \code{NULL}.}

\item{type}{Three different type of estimators can be selected: type \code{'0'} (local constant on all the covariates), type \code{'1'} (local linear smoother on all the covariates), type \code{'alpha'} (local polynomial smoother only on the direction of interest).}

\item{degree}{Degree of the local polynomial smoother in the direction of interest when using the estimator of type \code{'alpha'}. Defaults to \code{NULL} for the case when using estimators of type \code{'0'} or \code{'1'}.}

\item{qderivate}{If TRUE, it calculates \code{g^(q+1)/(q+1)!} for each component only for the type \code{'alpha'} method. Defaults to \code{FALSE}.}

\item{orderkernel}{Order of the kernel used in the nuisance directions when using the estimator of type \code{'alpha'}. Defaults to \code{2}.}

\item{Qmeasure}{A matrix of points where the integration procedure ocurrs. Defaults to \code{NULL} for calcuting the integrals over the sample.}
}
\value{
A list with the following components:
\item{mu}{Estimate for the intercept.}
\item{g.matrix}{Matrix of estimated additive components (n by p).}
\item{prediction }{Matrix of estimated additive components for the points listed in the argument point.}
\item{mul}{A vector of size p showing in each component the estimated intercept that considers only that direction of interest when using the type \code{'alpha'} method.}
\item{g.derivative }{Matrix of estimated derivatives of the additive components (only when qderivate is \code{TRUE}) (n by p).}
\item{prediction.derivate }{Matrix of estimated derivatives of the additive components for the points listed in the argument point (only when qderivate is \code{TRUE}).}
\item{Xp}{Matrix of explanatory variables.}
\item{yp}{Vector of responses.}
}
\description{
This function computes the standard marginal integration procedures for additive models.
}
\details{
This function computes three types of classical marginal integration procedures for additive models, that is, considering a squared loss function.
}
\examples{
function.g1 <- function(x1) 24*(x1-1/2)^2-2
function.g2 <- function(x2) 2*pi*sin(pi*x2)-4
n <- 150
x1 <- runif(n)
x2 <- runif(n)
X <- cbind(x1, x2)
eps <- rnorm(n,0,sd=0.15)
regresion <- function.g1(x1) + function.g2(x2)
y <- regresion + eps
bandw <- matrix(0.25,2,2)
set.seed(8090)
nQ <- 80 
Qmeasure <- matrix(runif(nQ*2), nQ, 2)
fit.cl <- margint.cl(Xp=X, yp=y, windows=bandw, type='alpha', degree=1, Qmeasure=Qmeasure)

}
\references{
Chen R., Hardle W., Linton O.B. and Severance-Lossin E. (1996). Nonparametric estimation of additive separable regression models. Physica-Verlag HD, Switzerland.
Linton O. and Nielsen J. (1995). A kernel method of estimating structured nonparametric regression based on marginal integration. Biometrika, 82(1), 93-101.
Severance-Lossin E. and Sperlich S. (1999). Estimation of derivatives for additive separable models. Statistics, 33(3), 241-265.
Tjostheim D. and Auestad B. (1994). Nonparametric identification of nonlinear time series: Selecting significant lags. Journal of the American Statistical Association, 89(428), 1410-1430.
}
\author{
Alejandra Martinez, \email{ale_m_martinez@hotmail.com}, Matias Salibian-Barrera
}
