% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictRisk.R
\name{predictRisk}
\alias{predictRisk}
\alias{predictRisk.CauseSpecificCox}
\alias{predictRisk.riskRegression}
\alias{predictRisk.FGR}
\alias{predictRisk.prodlim}
\alias{predictRisk.rfsrc}
\alias{predictRisk.aalen}
\alias{predictRisk.cox.aalen}
\alias{predictRisk.coxph}
\alias{predictRisk.cph}
\alias{predictRisk.default}
\alias{predictRisk.matrix}
\alias{predictRisk.pecCtree}
\alias{predictRisk.pecCforest}
\alias{predictRisk.psm}
\alias{predictRisk.selectCox}
\alias{predictRisk.survfit}
\alias{predictRisk.randomForest}
\alias{predictRisk.lrm}
\alias{predictRisk.glm}
\alias{predictRisk.rpart}
\title{Extrating predicting risks from regression models}
\usage{
\method{predictRisk}{glm}(object,newdata,...)
\method{predictRisk}{cox.aalen}(object,newdata,times,...)
\method{predictRisk}{cph}(object,newdata,times,...)
\method{predictRisk}{coxph}(object,newdata,times,...)
\method{predictRisk}{matrix}(object,newdata,times,cause,...)
\method{predictRisk}{selectCox}(object,newdata,times,...)
\method{predictRisk}{psm}(object,newdata,times,...)
\method{predictRisk}{survfit}(object,newdata,times,...)
\method{predictRisk}{riskRegression}(object,newdata,times,cause,...)
\method{predictRisk}{prodlim}(object,newdata,times,cause,...)
\method{predictRisk}{rfsrc}(object,newdata,times,cause,...)
\method{predictRisk}{FGR}(object,newdata,times,cause,...)
\method{predictRisk}{CauseSpecificCox}(object,newdata,times,cause,...)
}
\arguments{
\item{object}{A fitted model from which to extract predicted event
probabilities}

\item{newdata}{A data frame containing predictor variable combinations for
which to compute predicted event probabilities.}

\item{\dots}{Additional arguments that are passed on to the current method.}

\item{times}{A vector of times in the range of the response variable, for
which the cumulative incidences event probabilities are computed.}

\item{cause}{Identifies the cause of interest among the competing events.}
}
\value{
For binary outcome a vector with predicted risks. For survival outcome with and without
competing risks
a matrix with as many rows as \code{NROW(newdata)} and as many
columns as \code{length(times)}. Each entry is a probability and in
rows the values should be increasing.
}
\description{
Extract event probabilities from fitted regression models and machine learning objects.
}
\details{
The function predictRisk is a generic function, meaning that it invokes
specifically designed functions depending on the 'class' of the first
argument.

See \code{\link{predictRisk}}.


In uncensored binary outcome data there is no need to choose a time point.

When operating on models for survival analysis (without competing risks) the function still
predicts the risk, as 1 - S(t|X) where S(t|X) is survival chance of a subject characterized
by X.

When there are competing risks (and the data are right censored) one needs
to specify both the time horizon for prediction (can be a vector) and the
cause of the event. The function then extracts the absolute risks F_c(t|X)
aka the cumulative incidence of an event of type/cause c until time t for a
subject characterized by X. Depending on the model it may or not be possible
to predict the risk of all causes in a competing risks setting. For example. a
cause-specific Cox (CSC) object allows to predict both cases whereas a Fine-Gray regression
model (FGR) is specific to one of the causes.
}
\examples{
## binary outcome
library(rms)
set.seed(7)
x <- abs(rnorm(20))
d <- data.frame(y=rbinom(20,1,x/max(x)),x=x,z=rnorm(20))
nd <- data.frame(y=rbinom(8,1,x/max(x)),x=abs(rnorm(8)),z=rnorm(8))
fit <- lrm(y~x+z,d)
predictRisk(fit,newdata=nd)

## survival outcome
# generate survival data
library(prodlim)
set.seed(100)
d <- sampleData(100,outcome="survival")
d[,X1:=as.numeric(as.character(X1))]
d[,X2:=as.numeric(as.character(X2))]
# then fit a Cox model
library(rms)
cphmodel <- cph(Surv(time,event)~X1+X2,data=d,surv=TRUE,x=TRUE,y=TRUE)
# or via survival
library(survival)
coxphmodel <- coxph(Surv(time,event)~X1+X2,data=d,x=TRUE,y=TRUE)

# Extract predicted survival probabilities 
# at selected time-points:
ttt <- quantile(d$time)
# for selected predictor values:
ndat <- data.frame(X1=c(0.25,0.25,-0.05,0.05),X2=c(0,1,0,1))
# as follows
predictRisk(cphmodel,newdata=ndat,times=ttt)
predictRisk(coxphmodel,newdata=ndat,times=ttt)

# stratified cox model
sfit <- coxph(Surv(time,event)~strata(X1)+X2,data=d,x=TRUE,y=TRUE)
predictRisk(sfit,newdata=d[1:3,],times=c(1,3,5,10))

## simulate learning and validation data
learndat <- sampleData(100,outcome="survival")
valdat <- sampleData(100,outcome="survival")
## use the learning data to fit a Cox model
library(survival)
fitCox <- coxph(Surv(time,event)~X1+X2,data=learndat,x=TRUE,y=TRUE)
## suppose we want to predict the survival probabilities for all subjects
## in the validation data at the following time points:
## 0, 12, 24, 36, 48, 60
psurv <- predictRisk(fitCox,newdata=valdat,times=seq(0,60,12))
## This is a matrix with survival probabilities
## one column for each of the 5 time points
## one row for each validation set individual

# Do the same for a randomSurvivalForest model
# library(randomForestSRC)
# rsfmodel <- rfsrc(Surv(time,event)~X1+X2,data=learndat)
# prsfsurv=predictRisk(rsfmodel,newdata=valdat,times=seq(0,60,12))
# plot(psurv,prsfsurv)

## Cox with ridge option
f1 <- coxph(Surv(time,event)~X1+X2,data=learndat,x=TRUE,y=TRUE)
f2 <- coxph(Surv(time,event)~ridge(X1)+ridge(X2),data=learndat,x=TRUE,y=TRUE)
\dontrun{
plot(predictRisk(f1,newdata=valdat,times=10),
     riskRegression:::predictRisk.coxph(f2,newdata=valdat,times=10),
     xlim=c(0,1),
     ylim=c(0,1),
     xlab="Unpenalized predicted survival chance at 10",
     ylab="Ridge predicted survival chance at 10")
}

## competing risks

library(survival)
library(riskRegression)
library(prodlim)
train <- SimCompRisk(100)
test <- SimCompRisk(10)
cox.fit  <- CSC(Hist(time,cause)~X1+X2,data=train)
predictRisk(cox.fit,newdata=test,times=seq(1:10),cause=1)

## with strata
cox.fit2  <- CSC(list(Hist(time,cause)~strata(X1)+X2,Hist(time,cause)~X1+X2),data=train)
predictRisk(cox.fit2,newdata=test,times=seq(1:10),cause=1)

}
\seealso{
See \code{\link{predictRisk}}.
}
\author{
Thomas A. Gerds \email{tag@biostat.ku.dk}
}
\keyword{survival}
