% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/allinone.R
\encoding{UTF-8}
\name{allinone}
\alias{allinone}
\title{Apply all available (sequential) permutation testing approaches of variable importance measures with one function call}
\usage{
allinone(
  data,
  yname,
  Mmax = 500,
  varnames = NULL,
  p0 = 0.06,
  p1 = 0.04,
  alpha = 0.05,
  beta = 0.2,
  A = 0.1,
  B = 10,
  h = 8,
  nperm = 1,
  ntree = 500,
  progressbar = TRUE,
  condinf = FALSE,
  ...
)
}
\arguments{
\item{data}{A \code{data.frame} containing the variables in the model.}

\item{yname}{Name of outcome variable.}

\item{Mmax}{Maximum number of permutations used in each permutation test. Default is 500.}

\item{varnames}{Optional. Names of the variables for which testing should be performed. By default all variables in \code{data} with the exception of the outcome variable are used.}

\item{p0}{The value of the p-value in the null hypothesis (H0: p = p0) of SPRT and SAPT. Default is 0.06.}

\item{p1}{The value of the p-value in the alternative hypothesis (H1: p = p1) of SPRT and SAPT. Default is 0.04.}

\item{alpha}{The significance level of SPRT when p = p0. Also known as type I error. Default is 0.05.}

\item{beta}{One minus the power of SPRT when p = p1. Also known as type II error. Default is 0.2.}

\item{A}{The quantity A in the formula of SAPT. Default is 0.1 for a type I error of 0.05. Usually not changed by the user.}

\item{B}{The quantity B in the formula of SAPT. Default is 10 (1/A) for a type I error of 0.05. Usually not changed by the user.}

\item{h}{The quantity h in the formula for the sequential Monte Carlo p-value. The default value for h is 8. Larger values lead to more precise p-value estimates,
but are computationally more expensive.}

\item{nperm}{The numbers of permutations of the out-of-bag observations over which the results are averaged, when calculating the variable importance measure values. Default is 1. Larger values than 1 can only be considered when \code{condinf=TRUE}, that is, when using random forests
with conditional inference trees (Hothorn et al., 2006) as base learners.}

\item{ntree}{Number of trees per forest. Default is 500.}

\item{progressbar}{Output the current progress of the calculations for each variable to the console? Default is TRUE.}

\item{condinf}{Set this value to \code{TRUE} if random forests using conditional inference trees (Hothorn et al., 2006) should
be used and to \code{FALSE} if classical random forests using CART trees should be used. Default is \code{FALSE}.}

\item{...}{Further arguments passed to \code{ranger::ranger} (if \code{condinf=FALSE}) or \cr \code{party::cforest_unbiased()} (if \code{condinf=TRUE}).}
}
\value{
Object of class \code{allinone} with elements
  \item{\code{varimp}}{Variable importance for each considered independent variable.}
  \item{\code{testres}}{The results ("keep H0" vs. "accept H1") of the tests for each considered independent variable.}
  \item{\code{pvalues}}{The p-values of the tests for each considered independent variable. Note that p-values are only obtained for the
  method types "pval" and "complete".}
  \item{\code{stoppedearly}}{For each independent variable, whether the calculations stopped early ("yes") or the maximum of \code{Mmax} permutations was reached ("no").}
  \item{\code{perms}}{The number of permutations performed for each independent variable.}
  \item{\code{Mmax}}{Maximum number of permutations used in each permutation test.}
  \item{\code{ntree}}{Number of trees per forest.}
  \item{\code{comptime}}{The time the computations needed.}
}
\description{
This is a helper function, which allows to perform all (sequential) permutation testing approaches of variable importance measures described in \code{\link{rfvimptest}}
with a single function call. This may be useful for comparing the results obtained using the different approaches.
Importantly, this function is computationally efficient by re-using the permuted variable importance values obtained
for the conventional permutation test (that performs all \code{Mmax} permutations) for the other approaches. For details
on the different approaches see \code{\link{rfvimptest}}.
}
\examples{
\donttest{

# Load package:
library("rfvimptest")

# Set seed to obtain reproducible results:
set.seed(1234)

# Load example data:
data(hearth2)

# NOTE: For illustration purposes very small numbers of maximum
# permutations are considered in the below examples.
# This number would be much too small for actual applications.
# The default number is Max=500.

# When using condinf=FALSE (default) the results for the two-sample
# permutation tests are not obtained:
(ptest <- allinone(data=hearth2, yname="Class",  Mmax=20))

# Variable importance values with p-values from the Monte Carlo p-value
# and the complete approach:
ptest$varimp
ptest$pvalues$pval
ptest$pvalues$complete


# When setting condinf=TRUE the results are obtained for all approaches,
# that is, including those for the two-sample permutation tests
# (in this illustration very small number of trees ntree=30 are used,
# in practice much larger numbers should be used; the default is ntree=500):
(ptest_ci <- allinone(data=hearth2, yname="Class", condinf=TRUE, ntree=30, Mmax=10))
ptest_ci$testres

}

}
\references{
\itemize{
  \item Breiman, L. (2001). Random forests. Mach Learn, 45:5-32, <\doi{10.1023/A:1010933404324}>.
  \item Coleman, T., Peng, W., Mentch, L. (2019). Scalable and efficient hypothesis testing with random forests. arXiv preprint arXiv:1904.07830, <\doi{10.48550/arXiv.1904.07830}>.
  \item Hapfelmeier, A., Hornung, R., Haller, B. (in prep.) Sequential Permutation Testing of Random Forest Variable Importance Measures.
  \item Hapfelmeier, A., Ulm, K. (2013). A new variable selection approach using Random Forests. CSDA 60:50–69, <\doi{10.1016/j.csda.2012.09.020}>.
  \item Hapfelmeier, A., Hothorn, T., Ulm, K., Strobl, C. (2014). A new variable importance measure for random forests with missing data. Stat Comput 24:21–34, <\doi{10.1007/s11222-012-9349-1}>.
  \item Hothorn, T., Hornik, K., Zeileis, A. (2006). Unbiased Recursive Partitioning: A Conditional Inference Framework. J Comput Graph Stat 15(3):651–674, <\doi{10.1198/106186006X133933}>.
  \item Wright, M. N., Ziegler, A. (2017). ranger: A fast implementation of random forests for high dimensional data in C++ and R. J Stat Softw 77:1-17, <\doi{10.18637/jss.v077.i01}>.
  }
}
\seealso{
\code{\link{rfvimptest}}
}
\author{
Alexander Hapfelmeier, Roman Hornung
}
