% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/report_spec.r
\name{create_report}
\alias{create_report}
\title{Create a report}
\usage{
create_report(
  file_path = "",
  output_type = "TXT",
  orientation = "landscape",
  units = "inches",
  paper_size = "letter",
  missing = "",
  font = "fixed",
  font_size = NULL
)
}
\arguments{
\item{file_path}{The output path of the desired report. Either a full path or
a relative path is acceptable.  This parameter is not required to create the
report_spec object, but will be required to write the report.  In addition, 
the file extension is not required.  If the file extension is not supplied,
the \code{\link{write_report}} function will add a file extension based
on the \code{output_type} specified.}

\item{output_type}{The report output type.  Default is "TXT".  Valid
values are "TXT", "RTF", "PDF", and "HTML".}

\item{orientation}{The page orientation of the desired report.  Valid values
are "landscape" or "portrait".  The default page orientation is "landscape".}

\item{units}{Specifies the units of measurement.  This setting will 
indicate the units for columns widths, margins, paper size, and other 
measurements. Valid values are "inches" or "cm" (centimeters).  
Default value is "inches".}

\item{paper_size}{The expected paper size on which the report may be 
printed.  The \code{paper_size} will determine how much text can fit on
one page.  Valid values are "letter", "legal", "A4", and "RD4".  Default is 
"letter".  For the HTML output type, a paper size of "none" is also valid. 
That means the HTML will be generated in an unbounded manner as a typical
web page.}

\item{missing}{How to display missing values in the report.  Default is
to replace them with an empty string, which removes them from the report.
To display missing values as is, set the missing parameter to NULL.  To
replace missing values with a character string (such as ".", "-", or "<NA>")
pass the desired character string to the missing parameter.}

\item{font}{The font to use on the report.  The font specified will be
used for the entire report.  Valid values are "Courier", "Arial", "Times",
and "fixed".  The value of "fixed" will create a fixed-width, text style
report in Courier font.  The \code{font} parameter only applies to RTF reports
at this time.  The default value is "fixed".}

\item{font_size}{The size of the font to use on the report. The \code{font_size}
specified will be used for the entire report.  Valid values are 8, 10, 
and 12.  The \code{font_size} parameter only applies to RTF, PDF, and HTML 
output types. The default value is 10.}
}
\value{
A new report_spec object.
}
\description{
Creates a report shell to which you may add titles, footnotes, content, etc.
}
\details{
This function is the constructor for the report object.  The report
object contains information needed to create a report. The object is
defined as an S3 object, and has a class of 'report_spec'.

The report object holds information concerning report page size, orientation, 
titles, footnotes, page header, page footer, margins, and other options.  
Use the \code{\link{add_content}} function to add content to the report.  
The report may be written to a file using the \code{\link{write_report}} 
function.
}
\section{Report family of functions}{

The report is the primary container for report specifications.  The
following functions add additional specifications to the report object 
initialized with \code{create_report}.
\itemize{
  \item \code{\link{titles}} to add titles to the report.
  \item \code{\link{footnotes}} to add footnotes to the report.
  \item \code{\link{title_header}} to add a title header to the report.
  \item \code{\link{page_header}} to add a page header to the report. 
  \item \code{\link{page_footer}} to add a page_footer to the report. 
  \item \code{\link{add_content}} to add content to the report.
  \item \code{\link{options_fixed}} to set options for fixed-width output.
  \item \code{\link{add_content}} to add content to the report.
  \item \code{\link{write_report}} to write the report to the file system.
}

The report family of functions are pipe-friendly.  After creating the 
report, you may pipe the object to any of the above functions to append
additional options.

Note that PDF output currently only supports a fixed-width style report.
A variable-width report with a choice of fonts will be available in future
versions of the \strong{reporter} package.
}

\examples{
library(reporter)
library(magrittr)

# Create temp file path
tmp <- file.path(tempdir(), "airquality.txt")

# Prepare Data
dat <- airquality[sample(1:153, 15), ]
dat$Month <-  as.Date(paste0("1973-", dat$Month, "-01"))

# Define table
tbl <- create_table(dat, show_cols = c("Month", "Day", "Wind", "Temp", "Ozone")) \%>\% 
  titles("Table 9.6", "Air Quality Sample Report") \%>\% 
  column_defaults(width = .5) \%>\% 
  define(Month, format = "\%B", align = "left", width = 1) \%>\% 
  define(Temp, format = "\%.0f") \%>\% 
  footnotes("* New York, May to September 1973")

# Define report 
rpt <- create_report(tmp, orientation = "portrait",  missing = "-") \%>\% 
  add_content(tbl) 

# Write the report to the file system
write_report(rpt)

# Write the report to the console
writeLines(readLines(tmp, encoding = "UTF-8"))

#                      Table 9.6
#              Air Quality Sample Report
#      
#      Month           Day   Wind   Temp  Ozone
#      ----------------------------------------
#      July              8    6.3     92     97
#      July              9    5.7     92     97
#      August            1    6.9     81     39
#      July             23   11.5     82      -
#      June              9   13.8     90     71
#      July             12   14.3     73     10
#      July              4   10.9     84      -
#      May              31    7.4     76     37
#      September        30   11.5     68     20
#      June             25      8     75      -
#      June             28   11.5     80      -
#      August           18    7.4     76     23
#      June             20   10.3     76     13
#      July              1    4.1     84    135
#      May              23    9.7     61      4
#      
#      * New York, May to September 1973
}
\seealso{
\code{\link{create_table}}, \code{\link{create_text}}, and
\code{\link{create_plot}} functions
to create content for the report.

Other report: 
\code{\link{add_content}()},
\code{\link{footnotes}()},
\code{\link{options_fixed}()},
\code{\link{page_by}()},
\code{\link{page_footer}()},
\code{\link{page_header}()},
\code{\link{print.report_spec}()},
\code{\link{set_margins}()},
\code{\link{title_header}()},
\code{\link{titles}()},
\code{\link{write_report}()}
}
\concept{report}
