% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_genoprob.R
\name{plot_genoprob}
\alias{plot_genoprob}
\alias{plot.calc_genoprob}
\title{Plot genotype probabilities for one individual on one chromosome.}
\usage{
plot_genoprob(
  probs,
  map,
  ind = 1,
  chr = NULL,
  geno = NULL,
  color_scheme = c("gray", "viridis"),
  col = NULL,
  threshold = 0,
  swap_axes = FALSE,
  ...
)

\method{plot}{calc_genoprob}(x, ...)
}
\arguments{
\item{probs}{Genotype probabilities (as produced by \code{\link[=calc_genoprob]{calc_genoprob()}})
or allele dosages (as produced by \code{\link[=genoprob_to_alleleprob]{genoprob_to_alleleprob()}}).}

\item{map}{Marker map (a list of vectors of marker positions).}

\item{ind}{Individual to plot, either a numeric index or an ID.}

\item{chr}{Selected chromosome to plot; a single character string.}

\item{geno}{Optional vector of genotypes or alleles to be shown
(vector of integers or character strings)}

\item{color_scheme}{Color scheme for the heatmap (ignored if \code{col} is provided).}

\item{col}{Optional vector of colors for the heatmap.}

\item{threshold}{Threshold for genotype probabilities; only genotypes that achieve
this value somewhere on the chromosome will be shown.}

\item{swap_axes}{If TRUE, swap the axes, so that the genotypes are
on the x-axis and the chromosome position is on the y-axis.}

\item{...}{Additional graphics parameters passed to \code{\link[graphics:image]{graphics::image()}}.}

\item{x}{Genotype probabilities (as produced by
\code{\link[=calc_genoprob]{calc_genoprob()}}) or allele dosages (as produced by
\code{\link[=genoprob_to_alleleprob]{genoprob_to_alleleprob()}}). (For the S3 type plot
function, this has to be called \code{x}.)}
}
\value{
None.
}
\description{
Plot the genotype probabilities for one individual on one chromosome, as a heat map.
}
\section{Hidden graphics parameters}{

A number of graphics parameters can be passed via \code{...}. For
example, \code{hlines}, \code{hlines_col}, \code{hlines_lwd}, and \code{hlines_lty} to
control the horizontal grid lines. (Use \code{hlines=NA} to avoid
plotting horizontal grid lines.) Similarly \code{vlines}, \code{vlines_col},
\code{vlines_lwd}, and \code{vlines_lty} for vertical grid lines. You can
also use many standard graphics parameters like \code{xlab} and \code{xlim}.
These are not included as formal parameters in order to avoid
cluttering the function definition.
}

\examples{
# load data and calculate genotype probabilities
iron <- read_cross2(system.file("extdata", "iron.zip", package="qtl2"))
iron <- iron[,"2"] # subset to chr 2
map <- insert_pseudomarkers(iron$gmap, step=1)
pr <- calc_genoprob(iron, map, error_prob=0.002)

# plot the probabilities for the individual labeled "262"
#  (white = 0, black = 1)
plot_genoprob(pr, map, ind="262")

# change the x-axis label
plot_genoprob(pr, map, ind="262", xlab="Position (cM)")

# swap the axes so that the chromosome runs vertically
plot_genoprob(pr, map, ind="262", swap_axes=TRUE, ylab="Position (cM)")

# This is more interesting for a Diversity Outbred mouse example
\dontrun{
file <- paste0("https://raw.githubusercontent.com/rqtl/",
               "qtl2data/master/DOex/DOex.zip")
DOex <- read_cross2(file)
# subset to chr 2 and X and individuals labeled "232" and "256"
DOex <- DOex[c("232", "256"), c("2", "X")]
pr <- calc_genoprob(DOex, error_prob=0.002)
# plot individual "256" on chr 2 (default is to pick first chr in the probs)
plot_genoprob(pr, DOex$pmap, ind="256")

# omit states that never have probability >= 0.5
plot_genoprob(pr, DOex$pmap, ind="256", threshold=0.05)

# X chr male 232: just show the AY-HY genotype probabilities
plot_genoprob(pr, DOex$pmap, ind="232", chr="X", geno=paste0(LETTERS[1:8], "Y"))
# could also indicate genotypes by number
plot_genoprob(pr, DOex$pmap, ind="232", chr="X", geno=37:44)
# and can use negative indexes
plot_genoprob(pr, DOex$pmap, ind="232", chr="X", geno=-(1:36))

# X chr female 256: just show the first 36 genotype probabilities
plot_genoprob(pr, DOex$pmap, ind="256", chr="X", geno=1:36)

# again, can give threshold to omit genotypes whose probabilities never reach that threshold
plot_genoprob(pr, DOex$pmap, ind="256", chr="X", geno=1:36, threshold=0.5)

# can also look at the allele dosages
apr <- genoprob_to_alleleprob(pr)
plot_genoprob(apr, DOex$pmap, ind="232")
}

}
\seealso{
\code{\link[=plot_genoprobcomp]{plot_genoprobcomp()}}
}
