#' Search For Synonyms
#' 
#' \code{synonyms} - Search for synonyms that match term(s).
#' 
#' @param terms The terms to find synonyms for.  
#' @param return.list logical.  If \code{TRUE} returns the output for multiple 
#' synonyms as a list by search term rather than a vector.
#' @param multiwords logical.  IF \code{TRUE} retains vector elements that 
#' contain phrases (defined as having one or more spaces) rather than a single 
#' word.
#' @param report.null logical.  If \code{TRUE} reports the words that no match 
#' was found at the head of the output.
#' @return Returns a list of vectors or vector of possible words that match 
#' term(s).
#' @param synonym.frame A dataframe or environment containing a dataframe of positive/negative words and weights.
#' @rdname synonyms
#' @references The synonyms dictionary (see \code{\link[qdapDictionaries]{SYNONYM}}) was 
#' generated by web scraping the 
#' \href{http://dictionary.reverso.net/english-synonyms/}{Reverso Online Dictionary}.
#' The word list fed to \href{http://dictionary.reverso.net/english-synonyms/}{Reverso} 
#' is the unique words from the combination of \code{\link[qdapDictionaries]{DICTIONARY}} 
#' and \code{\link[qdapDictionaries]{labMT}}.
#' @export
#' @examples
#' \dontrun{
#' synonyms(c("the", "cat", "job", "environment", "read", "teach"))
#' head(syn(c("the", "cat", "job", "environment", "read", "teach"), 
#'     return.list = FALSE), 30)
#' syn(c("the", "cat", "job", "environment", "read", "teach"), multiwords = FALSE)
#'
#' ## User defined synonym lookup
#' syn_dat <- list(
#'     like = list(c("want", "desire"), c("love", "care")),
#'     show = list(c("reveal"), c("movie", "opera")),
#'     R = c("old friend", "statistics language")
#' )
#' 
#' synonyms_frame(syn_dat)
#' synonyms_frame(syn_dat, envir=FALSE)
#' syn(c("R", "show"), synonym.frame = syn_frame(syn_dat))
#' 
#' syns.hash <- syn_frame(syn_dat, prior.frame = SYNONYM)
#' syn(c("R", "show", "like", "robot"), synonym.frame = syns.hash)
#' }
synonyms <- function(terms, return.list = TRUE, 
    multiwords = TRUE, report.null = TRUE, 
    synonym.frame = qdapDictionaries::env.syn){

    out <- lapply(qdap_recoder(terms, envr=synonym.frame, 
        missing=NULL), function(z) {

        if (is.null(z)){
            return(NULL)
        } else {
            x <- rcst(z)
        }
        if (!multiwords){
            x <- lapply(x, function(y) {
                mults <- grepl("\\s", y)
                if (any(mults)){
                    y <- y[!mults]
                }
                return(y)
            })
        }
        return(x)
    })
    names(out) <- terms
    if (report.null & any(sapply(out, is.null))) {
        message("no match for the following:\n")
        message(paste(names(out)[sapply(out, is.null)], collapse = ", "))
        message("========================\n")
    }
    if (return.list) {
        unlist(out, recursive = FALSE)
    } else {
        outs2 <- unlist(out)
        names(outs2) <- NULL
        unique(outs2)
    }
}



#' @rdname synonyms
#' @export
syn <- synonyms

#' Search For Synonyms
#' 
#' \code{synonyms_frame} - Generate a synonym lookup environment 
#' for use with the \code{synonym.frame} argument in the \code{synonym} 
#' function.
#' 
#' @param synonym.list A named list of lists (or vectors) of synonyms.
#' @param envir logical.  If \code{TRUE} a lookup table (a dataframe within 
#' an environment) is produced rather than a data.frame.
#' @param prior.frame A prior synonyms data.frame in the format produced by \code{synonym_frame}
#' @export
#' @rdname synonyms
synonyms_frame <- function(synonym.list, envir = TRUE, prior.frame) {
    
    synonym.list <- lapply(synonym.list, function(x) {
        if(is.list(x)) {
            x
        } else {
            list(x)
        }
    })
    phase1 <- lapply(synonym.list, lapply, paste, collapse = ", ")
    phase2 <- mapply(seqs, phase1, lapply(phase1, function(x) 1:length(x))) 
    phase3 <- list2df(lapply(phase2, paste, collapse = " @@@@ "), 
        col2 = "word", col1 = "match.string")[2:1]
    phase3[] <- lapply(phase3, as.character)
    if (!missing(prior.frame)) {
        phase3 <- data.frame(rbind(phase3, 
            prior.frame[!prior.frame[, "word"] %in%  phase3[, "word"], ]
        ), stringsAsFactors = FALSE)
    }
    if (!envir) return(phase3)
    hash(phase3, mode.out="character")
}

#' @rdname synonyms
#' @export
syn_frame <- synonyms_frame

seqs <- function(x, y) sprintf("[%s]%s", y, x)

qdap_recoder <- function(x, envr, missing){                               
    x <- as.character(x)                                                         
    sapply(x, hash_look, USE.NAMES = FALSE, envir=envr, missing = missing)                       
}   


rcst <- function(x) {              
    y <- c(sapply(strsplit(x, "@@@@"), Trim))
    nms <- bracketXtract(y, "square")
    y <- bracketX(y, "square")
    names(y) <- paste0("def_", nms)
    lapply(lapply(y, strsplit, "\\,"), function(x){
        Trim(unlist(x))
    })
}
