% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/maxcover_model.r
\name{maxcover_model}
\alias{maxcover_model}
\alias{maxcover_model.Raster}
\alias{maxcover_model.SpatialPolygons}
\alias{maxcover_model.numeric}
\title{Prepare a maximum coverage prioritization problem}
\usage{
maxcover_model(x, ...)

\method{maxcover_model}{numeric}(x, budget, rij, locked_in = integer(),
  locked_out = integer(), ...)

\method{maxcover_model}{Raster}(x, features, budget, rij,
  locked_in = integer(), locked_out = integer(), ...)

\method{maxcover_model}{SpatialPolygons}(x, features, budget, rij,
  locked_in = integer(), locked_out = integer(), ...)
}
\arguments{
\item{x}{object specifying the planning units to use in the reserve design
  exercise and their corresponding cost. The possible types of objects are:

  \bold{\code{RasterLayer}}: planning units are the raster cells and cell
  values should contain the repsective cost. If \code{x} is a RasterLayer, it
  may be desirable to exlcude some planning units from the analysis, for
  example those outside the study area. To exclude planning units, set the
  cost for those raster cells to \code{NA}.

 \bold{\code{SpatialPolygonsDataFrame}}: polygons correspond to the planning
 units and there must be an attribute field named \code{cost}.

 \bold{\code{numeric vector}}: a vector of planning unit costs. In this case
 the spatial representation of the planning units is not provided. Therefore,
 the \code{feature} argument is ignored and the representation matrix
 \code{rij} is required.

 \bold{\code{MarxanData}}: a \code{MarxanData} object, from the \code{marxan}
  package (available at \url{https://github.com/jeffreyhanson/marxan}),
 specifying a Marxan reserve design problem,
 which is equivalent to the minimum set cover problem. The \code{marxan}
 package can create \code{MarxanData} objects from a variety of data
 sources, including the standard Marxan input files. Thus this is valuable
 for those wanted to transition directly from Marxan. Note that all other
 arguments are ignored if x is a \code{MarxanData} object.}

\item{...}{additional arguments passed on to methods}

\item{budget}{numeric; budget for reserve.}

\item{rij}{numeric matrix (optional); matrix of representation levels of
conservation features (rows) within planning units (columns). \code{rij}
can be a sparse matrix from the \code{slam} package (i.e. a
\code{\link[slam]{simple_triplet_matrix}}) or a normal base matrix object.
Alternatively, a data frame representation of this matrix with three
variables: feature index (\code{rij$feature}), planning unit index
(\code{rij$pu}), and corresponding representation level
(\code{rij$amount}). If this matrix is not provided it will be calculated
based on the planning units and RasterStack of conservation feature
distributions.}

\item{locked_in}{integer; indices of planning units to lock in to final
solution. For example, it may be desirable to lock in planning units
already within protected areas. Alternatively, if \code{x} is a
\code{RasterLayer}, then \code{locked_in} can be a
binary \code{RasterLayer} with 1s identifying the planning units to lock
in.}

\item{locked_out}{integer; indices of planning units to lock out of final
solution. For example, it may be desirable to lock in planning units that
are already heavily developed and therefore have little viable habitat.
Alternatively, if \code{x} is a \code{RasterLayer}, then \code{locked_out}
can be a binary \code{RasterLayer} with 1s identifying the planning units
to lock out.}

\item{features}{RasterStack object; the distribution of conservation
features. Missing values (i.e. \code{NA}s) can be used to indicate the
absence of a feature in a particular cell instead of explicitly setting
these cells to zero. If \code{x} is a Raster object then \code{features}
should be defined on the same raster template. If \code{x} is a
SpatialPolygonsDataFrame \code{features} will be summarize over the
polygons using \code{\link{summarize_features}}. Not required if
\code{rij} is provided.}
}
\value{
A \code{maxcover_model} object describing the prioritization
  problem to be solved. This is an S3 object consisting of a list with the
  following components:

\itemize{
  \item \code{cost}: numeric vector of planning unit costs
  \item \code{rij}: representation matrix
  \item \code{budget}: budget for reserve
  \item \code{locked_in}: indices of locked in planning units
  \item \code{locked_out}: indices of locked out planning units
  \item \code{included}: logical vector indicating which planning units are
    to be included in the analysis. If all units are to be included, this is
    single value (\code{TRUE}). Using a subset of planning units is only
    permitted if the \code{x} argument is provided as a RasterLayer object.
}
}
\description{
Specify a maximum coverage systematic conservation prioritization problem
from input data in a variety of formats. These are constructor functions for
\code{maxcover_model} objects which encapsulate prioritization problems in
a standardized format.
}
\details{
In the context of systematic reserve design, the maximum coverage
  problem seeks to find the set of planning units that maximizes the overall
  level of representation across a suite of conservation features, while
  keeping cost within a fixed budget. The cost is often either the area of
  the planning units or the opportunity cost of foregone commericial
  activities (e.g. from logging or agriculture). Representation level is
  typically given by the occupancy within each planning unit, however, some
  measure of abundance or probability of occurence may also be used.

  This problem is roughly the opposite of what the conservation planning
  software Marxan does.
}
\section{Methods (by class)}{
\itemize{
\item \code{numeric}: Numeric vector of costs

\item \code{Raster}: RasterLayer of planning units and corresponding
costs

\item \code{SpatialPolygons}: SpatialPolygonsData frame of planning units with
cost attribute
}}
\examples{
# 5x5 raster template
e <- raster::extent(0, 1, 0, 1)
r <- raster::raster(e, nrows = 5, ncols = 5, vals = 1)

# generate 4 random feature distributions
set.seed(419)
f <- raster::stack(r, r, r, r)
f[] <- sample(0:1, raster::nlayers(f) * raster::ncell(f), replace = TRUE)
f <- setNames(f, letters[1:raster::nlayers(f)])
# genrate cost layer
cost <- r
cost[] <- rnorm(raster::ncell(cost), mean = 100, sd = 10)
cost <- setNames(cost, "cost")

# prepare prioritization model with budget at 25\% of total cost
b_25 <- 0.25 * raster::cellStats(cost, "sum")
model <- maxcover_model(x = cost, features = f, budget = b_25)

# the representation matrix (rij) can be supplied explicitly,
# in which case the features argument is no longer required
rep_mat <- unname(t(f[]))
model <- maxcover_model(x = cost, rij = rep_mat, budget = b_25)

# cells can be locked in or out of the final solution
model <- maxcover_model(x = cost, features = f, budget = b_25,
                        locked_in = 6:10,
                        locked_out = 16:20)
# alternatively, binary rasters can be supplied indicating locked in/out
r_locked_in <- raster::raster(r)
r_locked_in[] <- 0
# lock in cells 6-10
r_locked_in[6:10] <- 1
model_raste_lock <- maxcover_model(x = cost, features = f, budget = b_25,
                                   locked_in = r_locked_in,
                                   locked_out = 16:20)

# if some cells are to be exlcuded, e.g. those outside study area, set
# the cost to NA for these cells.
cost_na <- cost
cost_na[6:10] <- NA
model_na <- maxcover_model(x = cost_na, features = f, budget = b_25)
# the model object now contains an included component specifying which
# cells are to be included
model_na$included
which(!model_na$included)
# note that the representation matrix now has fewer columns because
# the decision variables corresponding to exlcuded cells have been removed
model$rij
model_na$rij

# planning units can also be supplied as a SpatialPolygonsDataFrame object
# with cost stored as an attribute (x$cost). Typically the function takes
# longer to execute with polygons because summarizing features over planning
# units is less efficient.
cost_spdf <- raster::rasterToPolygons(cost)
model_spdf <- maxcover_model(x = cost_spdf, features = f, budget = b_25)
}
\seealso{
\code{\link{minsetcover_model}} for the minimum set cover problem.
  \code{\link{maxtargets_model}} for a modified version of the maximum
  coverage problem that maximizes attainment of representation targets.
}

