% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/POMDP.R
\name{POMDP}
\alias{POMDP}
\alias{O_}
\alias{T_}
\alias{R_}
\title{Define a POMDP Problem}
\usage{
POMDP(
  states,
  actions,
  observations,
  transition_prob,
  observation_prob,
  reward,
  discount = 0.9,
  horizon = Inf,
  terminal_values = NULL,
  start = "uniform",
  name = NA
)

O_(action = "*", end.state = "*", observation = "*", probability)

T_(action = "*", start.state = "*", end.state = "*", probability)

R_(action = "*", start.state = "*", end.state = "*", observation = "*", value)
}
\arguments{
\item{states}{a character vector specifying the names of the states. Note that
state names have to start with a letter.}

\item{actions}{a character vector specifying the names of the available actions.
Note that action names have to start with a letter.}

\item{observations}{a character vector specifying the names of the
observations. Note that observation names have to start with a letter.}

\item{transition_prob}{Specifies action-dependent transition probabilities
between states.  See Details section.}

\item{observation_prob}{Specifies the probability that an action/state
combination produces an observation.  See Details section.}

\item{reward}{Specifies the rewards structure dependent on action, states
and observations.  See Details section.}

\item{discount}{numeric; discount factor between 0 and 1.}

\item{horizon}{numeric; Number of epochs. \code{Inf} specifies an infinite
horizon.}

\item{terminal_values}{a vector with the terminal values for each state or a
matrix specifying the terminal rewards via a terminal value function (e.g.,
the alpha component produced by solve_POMDP).  A single 0 specifies that all
terminal values are zero.}

\item{start}{Specifies the initial probabilities for each state (i.e., the
initial belief), typically as a vector or the string \code{'uniform'}
(default).  This belief is used to calculate the total expected cumulative
reward. It is also used by some solvers. See Details section for more
information.}

\item{name}{a string to identify the POMDP problem.}

\item{action, start.state, end.state, observation, probability, value}{Values
used in the helper functions \code{O_()}, \code{R_()}, and \code{T_()} to
create an entry for \code{observation_prob}, \code{reward}, or
\code{transistion_prob} above, respectively. The default value \code{'*"'}
matches any action/state/observation.}
}
\value{
The function returns an object of class POMDP which is list of the model specification.
\code{\link[=solve_POMDP]{solve_POMDP()}} reads the object and adds a list element named
\code{'solution'}.
}
\description{
Defines all the elements of a POMDP problem including the discount rate, the
set of states, the set of actions, the set of observations, the transition
probabilities, the observation probabilities, and rewards.
}
\details{
In the following we use the following notation. The POMDP is a 7-duple:

\eqn{(S,A,T,R, \Omega ,O, \gamma)}.

\eqn{S} is the set of states; \eqn{A}
is the set of actions; \eqn{T} are the conditional transition probabilities
between states; \eqn{R} is the reward function; \eqn{\Omega} is the set of
observations; \eqn{O} are the conditional observation probabilities; and
\eqn{\gamma} is the discount factor. We will use lower case letters to
represent a member of a set, e.g., \eqn{s} is a specific state. To refer to
the size of a set we will use cardinality, e.g., the number of actions is
\eqn{|A|}.

\strong{Names used for mathematical symbols in code}
\itemize{
\item \eqn{S, s, s'}: \verb{'states', start.state', 'end.state'}
\item \eqn{A, a}: \verb{'actions', 'action'}
\item \eqn{\Omega, o}: \verb{'observations', 'observation'}
}

State names, actions and observations can be specified as strings or index numbers
(e.g., \code{start.state} can be specified as the index of the state in \code{states}).
For the specification as data.frames below, \code{'*'} can be used to mean
any  \code{start.state}, \code{end.state}, \code{action} or \code{observation}.

The specification below map to the format used by pomdp-solve
(see \url{http://www.pomdp.org}).

\strong{Specification of transition probabilities: \eqn{T(s' | s, a)}}

Transition probability to transition to state \eqn{s'} from given state \eqn{s}
and action \eqn{a}. The transition probabilities can be
specified in the following ways:
\itemize{
\item A data.frame with columns exactly like the arguments of \code{T_()}.
You can use \code{rbind()} with helper function \code{T_()} to create this data
frame.
\item A named list of matrices, one for each action. Each matrix is square with
rows representing start states \eqn{s} and columns representing end states \eqn{s'}.
Instead of a matrix, also the strings \code{'identity'} or \code{'uniform'} can be specified.
\item A function with the same arguments are \code{T_()}, but no default values
that returns the transition probability.
}

\strong{Specification of observation probabilities: \eqn{O(o | s', a)}}

The POMDP specifies the probability for each observation \eqn{o} given an
action \eqn{a} and that the system transitioned to the end state
\eqn{s'}. These probabilities can be specified in the
following ways:
\itemize{
\item A data frame with columns named exactly like the arguments of \code{O_()}.
You can use \code{rbind()}
with helper function \code{O_()} to create this data frame.
\item A named list of matrices, one for each action. Each matrix has
rows representing end states \eqn{s'} and columns representing an observation \eqn{o}.
Instead of a matrix, also the strings \code{'identity'} or \code{'uniform'} can be specified.
\item A function with the same arguments are \code{O_()}, but no default values
that returns the observation probability.
}

\strong{Specification of the reward function: \eqn{R(s, s', o, a)}}

The reward function can be specified in the following
ways:
\itemize{
\item A data frame with columns named exactly like the arguments of \code{R_()}.
You can use \code{rbind()}
with helper function \code{R_()} to create this data frame.
\item A list of lists. The list levels are \code{'action'} and \code{'start.state'}. The list elements
are matrices with
rows representing end states \eqn{s'} and columns representing an observation \eqn{o}.
\item A function with the same arguments are \code{R_()}, but no default values
that returns the reward.
}

\strong{Start Belief}

This belief is used to calculate the total expected cumulative reward
printed with the solved model.  The function \code{\link[=reward]{reward()}} can be
used to calculate rewards for any belief.

Some methods use this belief to decide which belief states to explore (e.g.,
the finite grid method).  The default initial belief is a uniform
distribution over all states. No initial belief state can be used by setting
\code{start = NULL}.

Options to specify the start belief state are:
\itemize{
\item A probability distribution over the states. That is, a vector
of \eqn{|S|} probabilities, that add up to \eqn{1}.
\item The string \code{"uniform"} for a uniform
distribution over all states.
\item An integer in the range \eqn{1} to \eqn{n} to specify the index of a single starting state.
\item a string specifying the name of a single starting state.
}

\strong{Time-dependent POMDPs}

Time dependence of transition probabilities, observation probabilities and
reward structure can be modeled by considering a set of episodes
representing epoch with the same settings. The length of each episode is
specified as a vector for \code{horizon}, where the length is the number of
episodes and each value is the length of the episode in epochs. Transition
probabilities, observation probabilities and/or reward structure can contain
a list with the values for each episode. See \code{\link[=solve_POMDP]{solve_POMDP()}} for
more details and an example.
}
\examples{
## Defining the Tiger Problem (it is also available via data(Tiger), see ? Tiger)

Tiger <- POMDP(
  name = "Tiger Problem",
  discount = 0.75,
  states = c("tiger-left" , "tiger-right"),
  actions = c("listen", "open-left", "open-right"),
  observations = c("tiger-left", "tiger-right"),
  start = "uniform",

  transition_prob = list(
    "listen" =     "identity",
    "open-left" =  "uniform",
    "open-right" = "uniform"
  ),

  observation_prob = list(
    "listen" = rbind(c(0.85, 0.15),
                     c(0.15, 0.85)),
    "open-left" =  "uniform",
    "open-right" = "uniform"
  ),

  # the reward helper expects: action, start.state, end.state, observation, value
  # missing arguments default to '*' matching any value.
  reward = rbind(
    R_("listen",                    v =   -1),
    R_("open-left",  "tiger-left",  v = -100),
    R_("open-left",  "tiger-right", v =   10),
    R_("open-right", "tiger-left",  v =   10),
    R_("open-right", "tiger-right", v = -100)
  )
)

Tiger

# Defining the Tiger problem using functions

trans_f <- function(action, start.state, end.state) {
  if(action == 'listen')
    if(end.state == start.state) return(1)
    else return(0)

  return(1/2) ### all other actions have a uniform distribution
}

obs_f <- function(action, end.state, observation) {
  if(action == 'listen')
    if(end.state == observation) return(0.85)
  else return(0.15)

  return(1/2)
}

rew_f <- function(action, start.state, end.state, observation) {
  if(action == 'listen') return(-1)
  if(action == 'open-left' && start.state == 'tiger-left') return(-100)
  if(action == 'open-left' && start.state == 'tiger-right') return(10)
  if(action == 'open-right' && start.state == 'tiger-left') return(10)
  if(action == 'open-right' && start.state == 'tiger-right') return(-100)
  stop('Not possible')
}

Tiger_func <- POMDP(
  name = "Tiger Problem",
  discount = 0.75,
  states = c("tiger-left" , "tiger-right"),
  actions = c("listen", "open-left", "open-right"),
  observations = c("tiger-left", "tiger-right"),
  start = "uniform",
  transition_prob = trans_f,
  observation_prob = obs_f,
  reward = rew_f
)

Tiger_func
}
\references{
pomdp-solve website: \url{http://www.pomdp.org}
}
\seealso{
Other POMDP: 
\code{\link{plot_belief_space}()},
\code{\link{sample_belief_space}()},
\code{\link{simulate_POMDP}()},
\code{\link{solve_POMDP}()},
\code{\link{solve_SARSOP}()},
\code{\link{transition_matrix}()},
\code{\link{update_belief}()},
\code{\link{write_POMDP}()}
}
\author{
Hossein Kamalzadeh, Michael Hahsler
}
\concept{POMDP}
