% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/conjointly.R
\name{conjointly}
\alias{conjointly}
\alias{expect_conjointly}
\alias{test_conjointly}
\title{Perform multiple rowwise validations for joint validity}
\usage{
conjointly(
  x,
  ...,
  .list = list2(...),
  preconditions = NULL,
  actions = NULL,
  brief = NULL,
  active = TRUE
)

expect_conjointly(
  object,
  ...,
  .list = list2(...),
  preconditions = NULL,
  threshold = 1
)

test_conjointly(
  object,
  ...,
  .list = list2(...),
  preconditions = NULL,
  threshold = 1
)
}
\arguments{
\item{x}{A data frame, tibble (\code{tbl_df} or \code{tbl_dbi}), or, an agent object of
class \code{ptblank_agent} that can be created with \code{\link[=create_agent]{create_agent()}}.}

\item{...}{a collection one-sided formulas that consist of validation step
functions that validate row units. Specifically, these functions should be
those with the naming pattern \verb{col_vals_*()}. An example of this is
\verb{~ col_vals_gte(., vars(a), 5.5), ~ col_vals_not_null(., vars(b)}).}

\item{.list}{Allows for the use of a list as an input alternative to \code{...}.}

\item{preconditions}{expressions used for mutating the input table before
proceeding with the validation. This is ideally as a one-sided R formula
using a leading \code{~}. In the formula representation, the \code{.} serves as the
input data table to be transformed (e.g.,
\code{~ . \%>\% dplyr::mutate(col = col + 10)}.}

\item{actions}{A list containing threshold levels so that the validation step
can react accordingly when exceeding the set levels. This is to be created
with the \code{\link[=action_levels]{action_levels()}} helper function.}

\item{brief}{An optional, text-based description for the validation step.}

\item{active}{A logical value indicating whether the validation step should
be active. If the step function is working with an agent, \code{FALSE} will make
the validation step inactive (still reporting its presence and keeping
indexes for the steps unchanged). If the step function will be operating
directly on data, then any step with \code{active = FALSE} will simply pass the
data through with no validation whatsoever. The default for this is \code{TRUE}.}

\item{object}{A data frame or tibble (\code{tbl_df} or \code{tbl_dbi}) that serves as
the target table for the expectation function.}

\item{threshold}{A simple failure threshold value for use with the
expectation function. By default, this is set to \code{1} meaning that any
single unit of failure in data validation results in an overall test
failure. Whole numbers beyond \code{1} indicate that any failing units up to
that absolute threshold value will result in a succeeding \strong{testthat}
test. Likewise, fractional values (between \code{0} and \code{1}) act as a
proportional failure threshold.}
}
\value{
For the validation function, the return value is either a
\code{ptblank_agent} object or a table object (depending on whether an agent
object or a table was passed to \code{x}). The expectation function invisibly
returns its input but, in the context of testing data, the function is
called primarily for its potential side-effects (e.g., signaling failure).
The test function returns a logical value.
}
\description{
The \code{conjointly()} validation function, the \code{expect_conjointly()} expectation
function, and the \code{test_conjointly()} test function all check whether test
units at each index (typically each row) all pass multiple validations with
\verb{col_vals_*()}-type functions. Because of the imposed constraint on the
allowed validation functions, all test units are rows of the table (after any
common \code{preconditions} have been applied). Each of the functions (composed
with multiple validation function calls) ultimately perform a rowwise test of
whether all sub-validations reported a \emph{pass} for the same test units. In
practice, an example of a joint validation is testing whether values for
column \code{a} are greater than a specific value while values for column \code{b} lie
within a specified range. The validation functions to be part of the conjoint
validation are to be supplied as one-sided \strong{R} formulas (using a leading
\code{~}, and having a \code{.} stand in as the data object). The validation function
can be used directly on a data table or with an \emph{agent} object (technically,
a \code{ptblank_agent} object) whereas the expectation and test functions can only
be used with a data table.
}
\details{
If providing multiple column names in any of the supplied validation step
functions, the result will be an expansion of sub-validation steps to that
number of column names. Aside from column names in quotes and in \code{vars()},
\strong{tidyselect} helper functions are available for specifying columns. They
are: \code{starts_with()}, \code{ends_with()}, \code{contains()}, \code{matches()}, and
\code{everything()}.

Having table \code{preconditions} means \strong{pointblank} will mutate the table just
before interrogation. Such a table mutation is isolated in scope to the
validation step(s) produced by the validation function call. Using
\strong{dplyr} code is suggested here since the statements can be translated to
SQL if necessary. The code is most easily supplied as a one-sided \strong{R}
formula (using a leading \code{~}). In the formula representation, the \code{.} serves
as the input data table to be transformed (e.g.,
\code{~ . \%>\% dplyr::mutate(col_a = col_b + 10)}). Alternatively, a function could
instead be supplied (e.g.,
\code{function(x) dplyr::mutate(x, col_a = col_b + 10)}).

Often, we will want to specify \code{actions} for the validation. This argument,
present in every validation function, takes a specially-crafted list
object that is best produced by the \code{\link[=action_levels]{action_levels()}} function. Read that
function's documentation for the lowdown on how to create reactions to
above-threshold failure levels in validation. The basic gist is that you'll
want at least a single threshold level (specified as either the fraction of
test units failed, or, an absolute value), often using the \code{warn_at}
argument. This is especially true when \code{x} is a table object because,
otherwise, nothing happens. For the \verb{col_vals_*()}-type functions, using
\code{action_levels(warn_at = 0.25)} or \code{action_levels(stop_at = 0.25)} are good
choices depending on the situation (the first produces a warning when a
quarter of the total test units fails, the other \code{stop()}s at the same
threshold level).

Want to describe this validation step in some detail? Keep in mind that this
is only useful if \code{x} is an \emph{agent}. If that's the case, \code{brief} the agent
with some text that fits. Don't worry if you don't want to do it. The
\emph{autobrief} protocol is kicked in when \code{brief = NULL} and a simple brief will
then be automatically generated.
}
\section{Function ID}{

2-14
}

\examples{
# For all examples here, we'll use
# a simple table with three numeric
# columns (`a`, `b`, and `c`); this is
# a very basic table but it'll be more
# useful when explaining things later
tbl <-
  dplyr::tibble(
    a = c(5, 2, 6),
    b = c(3, 4, 6),
    c = c(9, 8, 7)
  )
  
tbl
  
# A: Using an `agent` with validation
#    functions and then `interrogate()`

# Validate a number of things on a
# row-by-row basis using validation
# functions of the `col_vals*` type
# (all have the same number of test
# units): (1) values in `a` are less
# than `4`, (2) values in `c` are
# greater than the adjacent values in
# `a`, and (3) there aren't any NA
# values in `b`
agent <-
  create_agent(tbl = tbl) \%>\%
  conjointly(
    ~ col_vals_lt(., vars(a), 8),
    ~ col_vals_gt(., vars(c), vars(a)),
    ~ col_vals_not_null(., vars(b))
    ) \%>\%
  interrogate()
  
# Determine if this validation
# had no failing test units (there
# are 3 test units, one for each row)
all_passed(agent)

# Calling `agent` in the console
# prints the agent's report; but we
# can get a `gt_tbl` object directly
# with `get_agent_report(agent)`

# What's going on? Think of there being
# three parallel validations, each
# producing a column of `TRUE` or `FALSE`
# values (`pass` or `fail`) and line them
# up side-by-side, any rows with any
# `FALSE` values results in a conjoint
# `fail` test unit

# B: Using the validation function
#    directly on the data (no `agent`)

# This way of using validation functions
# acts as a data filter: data is passed
# through but should `stop()` if there
# is a single test unit failing; the
# behavior of side effects can be
# customized with the `actions` option
tbl \%>\%
  conjointly(
    ~ col_vals_lt(., vars(a), 8),
    ~ col_vals_gt(., vars(c), vars(a)),
    ~ col_vals_not_null(., vars(b))
  )

# C: Using the expectation function

# With the `expect_*()` form, we would
# typically perform one validation at a
# time; this is primarily used in
# testthat tests
expect_conjointly(
  tbl,
  ~ col_vals_lt(., vars(a), 8),
  ~ col_vals_gt(., vars(c), vars(a)),
  ~ col_vals_not_null(., vars(b))
)

# D: Using the test function

# With the `test_*()` form, we should
# get a single logical value returned
# to us
tbl \%>\%
  test_conjointly(
    ~ col_vals_lt(., vars(a), 8),
    ~ col_vals_gt(., vars(c), vars(a)),
    ~ col_vals_not_null(., vars(b))
  )

}
\seealso{
Other validation functions: 
\code{\link{col_exists}()},
\code{\link{col_is_character}()},
\code{\link{col_is_date}()},
\code{\link{col_is_factor}()},
\code{\link{col_is_integer}()},
\code{\link{col_is_logical}()},
\code{\link{col_is_numeric}()},
\code{\link{col_is_posix}()},
\code{\link{col_schema_match}()},
\code{\link{col_vals_between}()},
\code{\link{col_vals_equal}()},
\code{\link{col_vals_expr}()},
\code{\link{col_vals_gte}()},
\code{\link{col_vals_gt}()},
\code{\link{col_vals_in_set}()},
\code{\link{col_vals_lte}()},
\code{\link{col_vals_lt}()},
\code{\link{col_vals_not_between}()},
\code{\link{col_vals_not_equal}()},
\code{\link{col_vals_not_in_set}()},
\code{\link{col_vals_not_null}()},
\code{\link{col_vals_null}()},
\code{\link{col_vals_regex}()},
\code{\link{rows_distinct}()}
}
\concept{validation functions}
