% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot3logit-package.R
\docType{package}
\encoding{UTF-8}
\name{plot3logit-package}
\alias{plot3logit-package}
\title{Ternary Plots for Trinomial Regression Models}
\description{
An implementation of the ternary plot for interpreting regression
coefficients of trinomial regression models, as proposed in Santi, Dickson
and Espa (2018).
}
\details{
The package permits the covariate effects of trinomial regression models to
be represented graphically by means of a ternary plot. The aim of the
plots is helping the interpretation of regression coefficients in terms of
the effects that a change in regressors' values has on the probability
distribution of the dependent variable. Such changes may involve either a
single regressor, or a group of them (composite changes), and the package
permits both cases to be represented in a user-friendly way. Theoretical
and methodological details are illustrated and discussed in Santi, Dickson
and Espa (2018).

The package can read the results of \strong{both categorical and ordinal trinomial
logit} regression fitted by various functions (see the next section) and
creates a \code{field3logit} object which may be represented by means of functions
\code{\link{gg3logit}} and \code{\link{stat_3logit}}.

The \code{plot3logit} package inherits graphical classes and methods from the
package \code{\link[ggtern:ggtern-package]{ggtern}} (Hamilton 2018) which, in turn, is
based on the \code{\link[ggplot2:ggplot2-package]{ggplot2}} package (Wickham 2016).

Graphical representation based on \strong{standard graphics} is made available
through the package \code{\link[Ternary:TernaryPlot]{Ternary}} (Smith 2017) by
functions \code{\link{plot3logit}} and \code{\link{TernaryField}}, and in particular by the
\code{\link[=plot.field3logit]{plot}} method of \code{field3logit} objects.
}
\section{Compatibility}{

Function \code{\link{field3logit}} of package \code{plot3logit} can read trinomial
regression estimates from the output of the following functions:
\itemize{
\item \code{\link[nnet:multinom]{multinom}} of package \code{nnet} (logit regression);
\item \code{\link[MASS:polr]{polr}} of package \code{MASS} (ordinal logit regression);
\item \code{\link[mlogit:mlogit]{mlogit}} of package \code{mlogit} (logit regression).
}

Moreover, explicit matrix of regression coefficients can be passed to
\code{\link{field3logit}}. See examples and function \code{\link{field3logit}} for further
details.
}

\examples{
data(cross_1year)

# Read from "nnet::multinom"
library(nnet)
mod0 <- multinom(employment_sit ~ gender + finalgrade, data = cross_1year)
field0 <- field3logit(mod0, 'genderFemale')
gg3logit(field0) + stat_3logit()

# Read from "MASS::polr"
library(MASS)
mydata <- cross_1year
mydata$finalgrade <- factor(mydata$finalgrade,
  c('Low', 'Average', 'High'), ordered = TRUE)
mod1 <- polr(finalgrade ~ gender + irregularity, data = mydata)
field1 <- field3logit(mod1, 'genderFemale')
gg3logit(field1) + stat_3logit()

# Read from "mlogit::mlogit"
library(mlogit)
mydata <- mlogit.data(cross_1year, choice = 'employment_sit', shape = 'wide')
mod2 <- mlogit(employment_sit ~ 0 | gender + finalgrade, data = mydata)
field2 <- field3logit(mod2, 'genderFemale')
gg3logit(field2) + stat_3logit()

# Read from matrix
M <- matrix(c(-2.05, 0.46, -2.46, 0.37), nrow = 2)
rownames(M) <- c('(Intercept)', 'genderFemale')
attr(M, 'labs') <- c('Employed', 'Unemployed', 'Trainee')
field3 <- field3logit(M, c(0, 1))
gg3logit(field3) + stat_3logit()

}
\references{
Hamilton N. E., M. Ferry (2018) "\code{ggtern}: Ternary Diagrams Using ggplot2",
\emph{Journal of Statistical Software, Code Snippets}, \strong{87}(3), 1-17.
\doi{10.18637/jss.v087.c03}

Santi F., M. M. Dickson, G. Espa (2018) "A graphical tool for interpreting
regression coefficients of trinomial logit models", \emph{The American
Statistician}. \doi{10.1080/00031305.2018.1442368}

Smith M. R. (2017). "\code{Ternary}: An R Package for Creating Ternary Plots",
\emph{Zenodo}. \doi{10.5281/zenodo.1068996}

Wickham, H. (2016) \emph{\code{ggplot2}: Elegant Graphics for Data Analysis.}
Springer-Verlag, New York.
}
\seealso{
\code{\link{field3logit}}, \code{\link{gg3logit}}, \code{\link{TernaryField}}.
}
