# functions to aid in detecting linear dependend columns in the (transformed) 
# model matrix or estimated plm models:
#  * detect_lin_dep
#  * alias (the latter as a wrapper around alias.lm)
#
# doc file provides an extensive example how linear dependence can arise after
# the data transformation, e. g. for within transformation

### detect_lin_dep.matrix, .data.frame, .plm
detect_lin_dep.matrix <- function(object, suppressPrint = FALSE, ...) {
  if (!inherits(object, "matrix")) {
    stop("Input 'object' must be a matrix. Presumably, one wants a model matrix
         generated by some 'model.matrix' function.")}
  
  # do rank reduction to detect lin. dep. columns
  rank_rec <- sapply(1:ncol(object), function(col) qr(object[ , -col])$rank)
  
  if (diff(range(rank_rec)) == 0) {
    num <- NULL # return NULL if there is no linear dep.
  } else {
    num <- which(rank_rec == max(rank_rec))
    names(num) <- colnames(object)[num]
  }
  
  if(!suppressPrint) {
    if (is.null(num)) {
      print("No linear dependent column(s) detected.")
    } else {
      print(paste0("Suspiscious column number(s): ", paste(num,        collapse = ", ")))
      print(paste0("Suspiscious column name(s):   ", paste(names(num), collapse = ", ")))
    }
    return(invisible(num))
  }
  return(num)
}

detect_lin_dep.data.frame <- function(object, suppressPrint = FALSE, ...) {
  if (!inherits(object, "data.frame")) {
    stop("Input 'object' must be a data.frame")}

  return(detect_lin_dep.matrix(as.matrix(object), ...))
}

detect_lin_dep.plm <- function(object, suppressPrint = FALSE, ...) {
  if (!inherits(object, "plm")) {
    stop("Input 'object' must be of class \"plm\"")}

  return(detect_lin_dep.matrix(model.matrix(object), ...))
}


detect_lin_dep <- function(object, ...) {
  UseMethod("detect_lin_dep")
}

### alias.plm, alias.pFormula
# This is just a wrapper function to allow to apply the generic stats::alias on
# plm objects and pFormulas with the _transformed data_ (the transformed model.matrix).
# NB: arguments model and effect are not treated here.
alias.plm <- function(object, ...) {
  dots <- list(...)
  if (!is.null(dots$inst.method)) stop("alias.plm/alias.pFormula: IV not supported")
  if (length(formula(object))[2] == 2) stop("alias.plm/alias.pFormula: IV not supported")
  
  # catch unsupported alias.lm args and convert
  if (!is.null(dots$partial)) {
    if (dots$partial) {
      dots$partial <- FALSE
      warning("alias.plm/alias.pFormula: arg partial=\"TRUE\" not supported, changed to FALSE")
    }
  } 
  if (!is.null(dots$partial.pattern)) {
    if (dots$partial.pattern) {
      dots$partial.pattern <- FALSE
      warning("alias.plm/alias.pFormula: arg partial.pattern=\"TRUE\" not supported, changed to FALSE")
    }
  }
  
  X <- model.matrix(object)
  y <- pmodel.response(object)
  
  lm.fit.obj <- lm.fit(X, y)
  class(lm.fit.obj) <- "lm"
  lm.fit.obj$terms <- deparse(object$formula)
  
  ## use lm.fit rather than lm():
  ## could estimate lm model with lm(), but takes more resources and 
  ## need to remove extra classes "formula" for lm to prevent warning
  # form <- object$formula
  # form <- setdiff(class(form), c("pFormula", "Formula"))
  # Xdf <- as.data.frame(X)
  # ydf <- as.data.frame(y)
  # names(ydf) <- names(object$model)[1]
  # data <- cbind(ydf, Xdf)
  # lmobj <- lm(form, data = data)
  # return(stats::alias(lmobj))
  
  return(stats::alias(lm.fit.obj, ... = dots))
}

alias.pFormula <- function(object, data,
                           model = c("pooling", "within", "Between", "between",
                                     "mean", "random", "fd"),
                           effect = c("individual", "time", "twoways"),
                           ...) {
  dots <- list(...)
  if (!is.null(dots$inst.method)) stop("alias.plm/alias.pFormula: IV not supported")
  model <- match.arg(model)
  effect <- match.arg(effect)
  formula <- object

  # check if object is already pFormula, try to convert if not    
  if (!inherits(formula, "pFormula")) formula <- pFormula(formula)
  
  # check if data is already a model frame, convert to if not
  if (is.null(attr(data, "terms"))) {
    data <- model.frame.pFormula(pFormula(formula), data)
  }

  plmobj <- plm(formula, data = data, model = model, effect = effect, ...)
#  print(summary(plmobj))
  return(alias(plmobj, ...))
}

