% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_lmms.R
\name{fit_lmms}
\alias{fit_lmms}
\title{Step 1 of PRC-LMM: fits one linear mixed model for each response}
\usage{
fit_lmms(
  y.names,
  fixefs,
  ranefs,
  long.data,
  surv.data,
  t.from.base,
  n.boots = 0,
  n.cores = 1,
  verbose = TRUE
)
}
\arguments{
\item{y.names}{character vector with the names of the
response variables which the LMMs have to be fitted to}

\item{fixefs}{fixed effects formula for the model, example:
\code{~ time} (NB: do not include left-hand side!)}

\item{ranefs}{random effects formula for the model,
specified using the representation of random effect
structures of the \code{R} package \code{nlme}}

\item{long.data}{a data frame with the longitudinal predictors,
comprehensive of a variable called \code{id} with the subject
ids, and of a variable measuring time from baseline}

\item{surv.data}{a data frame with the survival data and (if
relevant) additional baseline, which should at least contain
a subject id (called \code{id}), the time to event outcome
(\code{time}), and binary event variable (\code{event})}

\item{t.from.base}{name of the variable containing time from
baseline in \code{long.data}}

\item{n.boots}{number of bootstrap samples; if 0, the random
effects are computed only for the original dataset and no
bootstrapping is performed}

\item{n.cores}{number of cores to use to parallelize the computation
of the cluster bootstrap procedure. If \code{ncores = 1} (default),
no parallelization is done. Tip: you can use \code{parallel::detectCores()}
to check how many cores are available on your computer}

\item{verbose}{if \code{TRUE} (default and recommended value), information
on the ongoing computations is printed in the console}
}
\value{
A list containing the following objects:
\itemize{
\item \code{call.info}: a list containing the following function
call information: \code{call}, \code{y.names}, \code{fixefs},
\code{ranefs};
\item \code{lmm.fits.orig}: a list with the LMMs fitted on the
original dataset (it should comprise as many LMMs as the elements
of \code{y.names} are);
\item \code{df.sanitized}: a sanitized version of the supplied
\code{long.data} dataframe, without the
longitudinal measurements that are taken after the event
or after censoring;
\item \code{n.boots} number of bootstrap samples;
\item \code{boot.ids} ids of bootstrapped subjects (when \code{n.boots > 0});
\item \code{lmms.fits.boot}: a list of lists, with the LMMs fitted
on eacj bootstrapped datasets (when \code{n.boots > 0}).
}
}
\description{
This function performs the first step for the estimation
of the PRC-LMM model proposed in Signorelli et al. (2020,
in review)
}
\examples{
# generate example data
set.seed(1234)
p = 4 # number of longitudinal predictors
simdata = simulate_prclmm_data(n = 100, p = p, p.relev = 2, 
             seed = 123, t.values = c(0, 0.2, 0.5, 1, 1.5, 2))
 
# specify options for cluster bootstrap optimism correction
# procedure and for parallel computing 
do.bootstrap = FALSE
# IMPORTANT: set do.bootstrap = TRUE to compute the optimism correction!
n.boots = ifelse(do.bootstrap, 100, 0)
parallelize = FALSE
# IMPORTANT: set parallelize = TRUE to speed computations up!
if (!parallelize) n.cores = 1
if (parallelize) {
   # identify number of available cores on your machine
   n.cores = parallel::detectCores()
   if (is.na(n.cores)) n.cores = 1
}

# step 1 of PRC-LMM: estimate the LMMs
y.names = paste('marker', 1:p, sep = '')
step1 = fit_lmms(y.names = y.names, 
                 fixefs = ~ age, ranefs = ~ age | id, 
                 long.data = simdata$long.data, 
                 surv.data = simdata$surv.data,
                 t.from.base = t.from.base,
                 n.boots = n.boots, n.cores = n.cores)
}
\references{
Signorelli, M., Spitali, P., Tsonaka, R. (in review).
On the prediction of survival outcomes using longitudinal
and high-dimensional (omic) data.
}
\seealso{
\code{\link{simulate_prclmm_data}},
\code{\link{summarize_lmms}} (step 2),
\code{\link{fit_prclmm}} (step 3),
\code{\link{performance_prclmm}}
}
\author{
Mirko Signorelli
}
