% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/opt.R
\name{pedmod_profile}
\alias{pedmod_profile}
\title{Computes Profile Likelihood Based Confidence Intervals}
\usage{
pedmod_profile(
  ptr,
  par,
  delta,
  maxvls,
  minvls = -1L,
  alpha = 0.05,
  abs_eps,
  rel_eps,
  which_prof,
  indices = NULL,
  maxvls_start = max(100L, as.integer(ceiling(maxvls/5))),
  minvls_start = if (minvls < 0) minvls else minvls/5,
  do_reorder = TRUE,
  use_aprx = FALSE,
  n_threads = 1L,
  cluster_weights = NULL,
  method = 0L,
  seed = 1L,
  verbose = FALSE,
  max_step = 15L,
  standardized = FALSE,
  use_tilting = FALSE,
  vls_scales = NULL,
  ...
)
}
\arguments{
\item{ptr}{object from \code{\link{pedigree_ll_terms}} or
\code{\link{pedigree_ll_terms_loadings}}.}

\item{par}{numeric vector with the maximum likelihood estimator e.g. from
\code{\link{pedmod_opt}}.}

\item{delta}{numeric scalar with an initial step to take. Subsequent steps
are taken by \code{2^(<iteration number> - 1) * delta}. Two times the
standard error is a good value or a guess thereof. Hessian approximations are
not implemented as of this writing and therefore the user needs to provide
some guess.}

\item{maxvls}{maximum number of samples in the approximation for each
marginal likelihood term.}

\item{minvls}{minimum number of samples for each
marginal likelihood term. Negative values provides a
default which depends on the dimension of the integration.}

\item{alpha}{numeric scalar with the confidence level required.}

\item{abs_eps}{absolute convergence threshold for
\code{\link{eval_pedigree_ll}} and \code{\link{eval_pedigree_grad}}.}

\item{rel_eps}{rel_eps convergence threshold for
\code{\link{eval_pedigree_ll}} and \code{\link{eval_pedigree_grad}}.}

\item{which_prof}{integer scalar with index of the parameter which the
profile likelihood curve should be computed for.}

\item{indices}{zero-based vector with indices of which log marginal
likelihood terms to include. Use \code{NULL} if all indices should be
used.}

\item{maxvls_start, minvls_start}{number of samples to use when finding the
initial values for the optimization.}

\item{do_reorder}{\code{TRUE} if a heuristic variable reordering should
be used. \code{TRUE} is likely the best value.}

\item{use_aprx}{\code{TRUE} if a less precise approximation of
\code{\link{pnorm}} and \code{\link{qnorm}} should be used. This may
reduce the computation time while not affecting the result much.}

\item{n_threads}{number of threads to use.}

\item{cluster_weights}{numeric vector with weights for each cluster. Use
\code{NULL} if all clusters have weight one.}

\item{method}{integer with the method to use. Zero yields randomized Korobov
lattice rules while one yields scrambled Sobol sequences.}

\item{seed}{seed to pass to \code{\link{set.seed}} before each gradient and
function evaluation. Use \code{NULL} if the seed should not be fixed.}

\item{verbose}{logical for whether output should be printed to the console
during the estimation of the profile likelihood curve.}

\item{max_step}{integer scalar with the maximum number of steps to take in
either directions.}

\item{standardized}{logical for whether to use the standardized or direct
parameterization. See \code{\link{standardized_to_direct}} and the vignette
at \code{vignette("pedmod", package = "pedmod")}.}

\item{use_tilting}{\code{TRUE} if the minimax tilting method suggested
by Botev (2017) should be used. See \doi{10.1111/rssb.12162}.}

\item{vls_scales}{can be a numeric vector with a positive scalar for each
cluster. Then \code{vls_scales[i] * minvls} and
\code{vls_scales[i] * maxvls} is used for cluster \code{i} rather than
\code{minvls} and \code{maxvls}. Set \code{vls_scales = NULL} if the latter
should be used.}

\item{...}{arguments passed on to \code{\link{pedmod_opt}}.}
}
\value{
A list with the following elements:
  \item{confs}{2D numeric vector with the profile likelihood based confidence
               interval.}
  \item{xs}{the points at which the profile likelihood is evaluated.}
  \item{p_log_Lik}{the log profile likelihood values at \code{xs}.}
  \item{data}{list with the returned objects from \code{\link{pedmod_opt}}.}
}
\description{
Computes likelihood ratio based confidence intervals for one the parameters
in the model.
}
\examples{
\donttest{
# we simulate outcomes with an additive genetic effect. The kinship matrix is
# the same for all families and given by
K <- matrix(c(
  0.5  , 0    , 0.25 , 0   , 0.25 , 0   , 0.125 , 0.125 , 0.125 , 0.125 ,
  0    , 0.5  , 0.25 , 0   , 0.25 , 0   , 0.125 , 0.125 , 0.125 , 0.125 ,
  0.25 , 0.25 , 0.5  , 0   , 0.25 , 0   , 0.25  , 0.25  , 0.125 , 0.125 ,
  0    , 0    , 0    , 0.5 , 0    , 0   , 0.25  , 0.25  , 0     , 0     ,
  0.25 , 0.25 , 0.25 , 0   , 0.5  , 0   , 0.125 , 0.125 , 0.25  , 0.25  ,
  0    , 0    , 0    , 0   , 0    , 0.5 , 0     , 0     , 0.25  , 0.25  ,
  0.125, 0.125, 0.25 , 0.25, 0.125, 0   , 0.5   , 0.25  , 0.0625, 0.0625,
  0.125, 0.125, 0.25 , 0.25, 0.125, 0   , 0.25  , 0.5   , 0.0625, 0.0625,
  0.125, 0.125, 0.125, 0   , 0.25 , 0.25, 0.0625, 0.0625, 0.5   , 0.25  ,
  0.125, 0.125, 0.125, 0   , 0.25 , 0.25, 0.0625, 0.0625, 0.25  , 0.5
), 10)

# simulates a data set.
#
# Args:
#   n_fams: number of families.
#   beta: the fixed effect coefficients.
#   sig_sq: the scale parameter.
sim_dat <- function(n_fams, beta = c(-1, 1, 2), sig_sq = 3){
  # setup before the simulations
  Cmat <- 2 * K
  n_obs <- NROW(K)
  Sig <- diag(n_obs) + sig_sq * Cmat
  Sig_chol <- chol(Sig)

  # simulate the data
  out <- replicate(
    n_fams, {
      # simulate covariates
      X <- cbind(`(Intercept)` = 1, Continuous = rnorm(n_obs),
                 Binary = runif(n_obs) > .5)

      # assign the linear predictor + noise
      eta <- drop(X \%*\% beta) + drop(rnorm(n_obs) \%*\% Sig_chol)

      # return the list in the format needed for the package
      list(y = as.numeric(eta > 0), X = X, scale_mats = list(Cmat))
    }, simplify = FALSE)

  # add attributes with the true values and return
  attributes(out) <- list(beta = beta, sig_sq = sig_sq)
  out
}

# simulate the data
set.seed(1)
dat <- sim_dat(100L)

# fit the model
ptr <- pedigree_ll_terms(dat, max_threads = 1L)
start <- pedmod_start(ptr = ptr, data = dat, n_threads = 1L)
fit <- pedmod_opt(ptr = ptr, par = start$par, n_threads = 1L, use_aprx = TRUE,
                  maxvls = 5000L, minvls = 1000L, abs_eps = 0, rel_eps = 1e-3)
fit$par # the estimate

# 90\% likelihood ratio based confidence interval for the log of the scale
# parameter
prof_out <- pedmod_profile(ptr = ptr, fit$par, delta = .4, maxvls = 5000L,
                           minvls = 1000L, alpha = .1, which_prof = 4L,
                           abs_eps = 0, rel_eps = 1e-3, verbose = TRUE)
exp(prof_out$confs) # the confidence interval

# plot the log profile likelihood
plot(exp(prof_out$xs), prof_out$p_log_Lik, pch = 16,
     xlab = expression(sigma), ylab = "log profile likelihood")
abline(v = exp(prof_out$confs), lty = 2)
}

}
\seealso{
\code{\link{pedmod_opt}}, \code{\link{pedmod_sqn}},
\code{\link{pedmod_profile_prop}}, and \code{\link{pedmod_profile_nleq}}
}
