% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tides.R
\name{webtide}
\alias{webtide}
\title{Get a Tidal Prediction from a WebTide Database}
\source{
The WebTide software may be downloaded for free at the
Department of Fisheries and Oceans (Canada) website at
\verb{http://www.bio.gc.ca/science/research-recherche/ocean/webtide/index-en.php}
(checked February 2016 and May 2017).
}
\usage{
webtide(
  action = c("map", "predict"),
  longitude,
  latitude,
  node,
  time,
  basedir = getOption("webtide"),
  region = "nwatl",
  plot = TRUE,
  tformat,
  debug = getOption("oceDebug"),
  ...
)
}
\arguments{
\item{action}{An indication of the action, either \code{action="map"} to
draw a map or \code{action="predict"} to get a prediction; see
\sQuote{Details}.}

\item{longitude, latitude}{optional location at which prediction is required (ignored if
\code{node} is given).}

\item{node}{optional integer relating to a node in the database. If \code{node}
is given, then neither \code{latitude} nor \code{longitude} may be given.
If \code{node} is positive, then specifies indicates the node. If it is negative,
\code{\link[=locator]{locator()}} is called so that the user can click (once) on the map, after
which the node is displayed on the map.}

\item{time}{a vector of times (in the UTC timezone)
at which prediction is to be made.
If not supplied, this will be the week starting at the present time,
computed with \code{\link[=presentTime]{presentTime()}}, with a 15 minute increment.}

\item{basedir}{directory containing the \code{WebTide} application.}

\item{region}{database region, given as a directory name in the WebTide
directory.  For example, \code{h3o} is for Halifax Harbour, \code{nwatl} is
for the northwest Atlantic, and \code{sshelf} is for the Scotian Shelf and
Gulf of Maine.}

\item{plot}{boolean indicating whether to plot.}

\item{tformat}{optional argument passed to \code{\link[=oce.plot.ts]{oce.plot.ts()}}, for
plot types that call that function.  (See \code{\link[=strptime]{strptime()}} for the
format used.)}

\item{debug}{an integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed. If one function calls another, it usually reduces the value of
\code{debug} first, so that a user can often obtain deeper debugging
by specifying higher \code{debug} values.}

\item{\dots}{optional arguments passed to plotting functions. A common
example is to set \code{xlim} and \code{ylim}, to focus a map region.}
}
\value{
The value depends on \code{action}:
\itemize{
\item If \code{action="map"} the return value is a
list containing the index of the nearest node, along with the
\code{latitude} and \code{longitude} of that node.  If
\code{plot} is \code{FALSE}, this value is returned invisibly.
\item If \code{action="predict"}, the return value is a list containing a vector
of times (\code{time}), as well as vectors of the predicted \code{elevation}
in metres and the predicted horizontal components of velocity, \code{u} and
\code{v}, along with the \code{node} number, and the \code{basedir} and
\code{region} as supplied to this function. If \code{plot} is \code{FALSE},
this value is returned invisibly.
}
}
\description{
Get a tidal prediction from a WebTide database. This only
works if the standalone WebTide application is installed,
and if it is installed in a standard location. The details
of installation are not within the oce purview.
}
\details{
There are two methods of using this function.
\emph{Case 1:} \code{action="map"}. In this case, if
\code{plot} is \code{FALSE}, a list is returned, containing
all the \code{node}s in the selected database, along with all
the \code{latitude}s and \code{longitude}s. This value is
also returned (silently) if \code{plot} is true, but in that case,
a plot is drawn to indicate the node locations. If \code{latitude} and
\code{longitude} are given, then the node nearest that spot is indicated on
the map; otherwise, if \code{node} is given, then the location of that
node is indicated. There is also a special case: if \code{node} is negative
and \code{interactive()} is \code{TRUE},
then \code{\link[=locator]{locator()}} is called, and the node nearest the spot
where the user clicks the mouse is indicated in the plot and in the
return value.

\emph{Case 2:} \code{action="predict"}. If \code{plot} is \code{FALSE},
then a list is returned, indicating \code{time}, predicted
\code{elevation}, velocity components \code{u} and \code{v},
\code{node} number, the name of the \code{basedir}, and
the \code{region}. If \code{plot} is \code{TRUE}, this list is returned
silently, and time-series plots are drawn for elevation, u, and v.

Naturally, \code{webtide} will not work unless WebTide has been installed on
the computer.
}
\section{Caution}{

WebTide is not an open-source application, so the present function was
designed based on little more than guesses about the WebTide file structure.
Users should be on the lookout for odd results.
}

\examples{
\dontrun{
## needs WebTide at the system level
library(oce)
## 1. prediction at Halifax NS
longitude <- -63.57
latitude <- 44.65
prediction <- webtide("predict", longitude=longitude, latitude=latitude)
mtext(sprintf("prediction at \%fN \%fE", latitude, longitude), line=0.75, side=3)
## 2. map
webtide(lon=-63.57,lat=44.65,xlim=c(-64,-63),ylim=c(43.0,46))
}

}
\seealso{
Other things related to tides: 
\code{\link{[[,tidem-method}},
\code{\link{[[<-,tidem-method}},
\code{\link{as.tidem}()},
\code{\link{plot,tidem-method}},
\code{\link{predict.tidem}()},
\code{\link{summary,tidem-method}},
\code{\link{tidalCurrent}},
\code{\link{tidedata}},
\code{\link{tidem-class}},
\code{\link{tidemAstron}()},
\code{\link{tidemVuf}()},
\code{\link{tidem}}
}
\author{
Dan Kelley
}
\concept{things related to tides}
