% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/argo.R
\name{read.argo}
\alias{read.argo}
\title{Read an Argo Data File}
\usage{
read.argo(
  file,
  encoding = NA,
  debug = getOption("oceDebug"),
  processingLog,
  ...
)
}
\arguments{
\item{file}{A character string giving the name of the file to load.}

\item{encoding}{ignored.}

\item{debug}{A flag that turns on debugging.  Set to 1 to get a moderate amount
of debugging information, or to 2 to get more.}

\item{processingLog}{If provided, the action item to be stored in the log.
(Typically only provided for internal calls; the default that it provides is
better for normal calls by a user.)}

\item{...}{additional arguments, passed to called routines.}
}
\value{
An \linkS4class{argo} object.
}
\description{
\code{read.argo} is used to read an Argo file, producing an \linkS4class{argo} object.
The file must be in the ARGO-style NetCDF format described
in the Argo documentation (see references 2 and 3).
}
\details{
See the Argo documentation (see references 2 and 3) for some details on what files contain.
Many items listed in section 2.2.3 of reference 3 are read from the
file and stored in the \code{metadata} slot, with the exception of
\code{longitude} and \code{latitude}, which are stored in the
\code{data} slot, alongside hydrographic information.

The names of several data parameters stored within the netCDF file
are altered to fit the oce context. For example, \code{PRES} becomes \code{pressure},
matching the name of this variable in other oce data types.
The original names are reported by \verb{summary,argo-method}, and
data may be extracted with \verb{[[,argo-method} using those names, so
the renaming should not be too inconvenient to Argo experts who
are new to oce.

Several of the netCDF global attributes are also renamed before
placement in the \code{metadata} slot of the return value.  These include
\code{conventions}, \code{featureType}, \code{history}, \code{institution},
\code{nParameters}, \code{nProfiles},  \code{references}, \code{source}, \code{title},
and \code{userManualVersion}.
These names are derived from those in the netcdf
file, and mainly follow the pattern explained in the
\dQuote{Variable renaming convention} section.

For profile data (as indicated by the NetCDF global attribute
named \code{"featureType"} being equal to \code{"trajectoryProfile"}),
the NetCDF item named \code{"STATION_PARAMETERS"} controls
whether variables in the source file will be stored in the
\code{metadata} or \code{data} slot of the returned object.
If \code{STATION_PARAMETERS} is not present, as is the case for
trajectory files (which are detected by \code{featureType} being
\code{"trajectory"}), some guesses are made as to what goes in
\code{data} and \code{metadata} slots.

Each data item can have variants, as
described in Sections 2.3.4 of reference 3.
For example, if \code{"PRES"} is found in \code{STATION_PARAMETERS},
then \code{PRES} (pressure) data are sought in the file, along with
\code{PRES_QC}, \code{PRES_ADJUSTED}, \code{PRES_ADJUSTED_QC}, and
\code{PRES_ERROR}. The same pattern works for other profile data. The variables
are stored with names created as explained in the
\dQuote{Variable renaming convention} section below. Note that
flags, which are stored variables ending in \code{"_QC"} in the netcdf
file, are stored in the \code{flags} item within the \code{metadata} slot
of the returned object; thus, for example,
\code{PRES_QC} is stored as \code{pressure} in \code{flags}.
}
\section{Variable renaming convention}{

Argo netcdf files employ a \code{"SNAKE_CASE"} naming scheme (sometimes
using lower case) that is inconsistent with the \code{"camelCase"} scheme
used in oce. Since argo objects are just a small part of oce, a decision
was made to rename argo items. For example, \code{"CYCLE_NUMBER"} in the netcdf file
becomes \code{"cycleNumber"} in the oce object returned by \code{read.argo}.
(Note that \verb{[[,argo-method} also accepts \code{"cycle"} for this item.)
The conversion for objects in the \code{data} slot often also involves
expanding on argo abbreviations, e.g. \code{"PSAL"} becomes \code{"salinity"}.
The renaming work is carried out with \code{\link[=argoNames2oceNames]{argoNames2oceNames()}} for
handles both name expansion for several dozen special cases,
and with \code{\link[=snakeToCamel]{snakeToCamel()}} with the \code{specialCase} argument
set to \code{"QC"}. While this results in variable names that should make
sense in the general oce context (where, for example, salinity is expected
to be stored in a variable named \code{"salinity"}), it may be confusing
to argo experts who are just starting to use oce.  Such people might
find it helpful to use e.g. \code{sort(names(x[["metadata"]]))} to get a list
of all items in the \code{metadata} slot (or similar with \code{"data"}), since working
in reverse may be easier than simply guessing at what names oce has chosen.
(Note that prior to 2020 June 24, some metadata items were stored in
\code{"SNAKE_CASE"}.)
}

\section{Data sources}{

Argo data are made available at several websites. A bit of detective
work can be required to track down the data.

Some servers provide data for floats that surfaced in a given ocean
on a given day, the anonymous FTP server
\code{usgodae.org/pub/outgoing/argo/geo/} being an example.

Other servers provide data on a per-float basis. A complicating
factor is that these data tend to be categorized by "dac" (data
archiving centre), which makes it difficult to find a particular
float. For example,
\code{https://www.usgodae.org/ftp/outgoing/argo/} is the top level of
a such a repository. If the ID of a float is known but not the
"dac", then a first step is to download the text file
\code{https://www.usgodae.org/ftp/outgoing/argo/ar_index_global_meta.txt}
and search for the ID. The first few lines of that file are header,
and after that the format is simple, with columns separated by slash
(\code{/}). The dac is in the first such column and the float ID in the
second. A simple search will reveal the dac.
For example \code{data(argo)} is based on float 6900388, and the line
containing that token is
\verb{bodc/6900388/6900388_meta.nc,846,BO,20120225005617}, from
which the dac is seen to be the British Oceanographic Data Centre
(\code{bodc}). Armed with that information, visit
\code{https://www.usgodae.org/ftp/outgoing/argo/dac/bodc/6900388}
and see a directory called \code{profiles} that contains a NetCDF
file for each profile the float made. These can be read with
\code{read.argo}. It is also possible, and probably more common,
to read a NetCDF file containing all the profiles together and for
that purpose the file
\code{https://www.usgodae.org/ftp/outgoing/argo/dac/bodc/6900388/6900388_prof.nc}
should be downloaded and provided as the \code{file} argument to
\code{read.argo}.  This can be automated as in Example 2,
although readers are cautioned that URL structures tend to change
over time.

Similar steps can be followed on other servers.
}

\examples{
\dontrun{
## Example 1: read from a local file
library(oce)
d <- read.argo("/data/OAR/6900388_prof.nc")
summary(d)
plot(d)

## Example 2: construct URL for download (brittle)
id <- "6900388"
url <- "https://www.usgodae.org/ftp/outgoing/argo"
if (!length(list.files(pattern="argo_index.txt")))
    download.file(paste(url, "ar_index_global_meta.txt", sep="/"), "argo_index.txt")
index <- readLines("argo_index.txt")
line <- grep(id, index)
if (0 == length(line)) stop("id ", id, " not found")
if (1 < length(line)) stop("id ", id, " found multiple times")
dac <- strsplit(index[line], "/")[[1]][1]
profile <- paste(id, "_prof.nc", sep="")
float <- paste(url, "dac", dac, id, profile, sep="/")
download.file(float, profile)
argo <- read.argo(profile)
summary(argo)
}


}
\references{
\enumerate{
\item \verb{https://argo.ucsd.edu}
\item Argo User's Manual Version 3.2, Dec 29th, 2015, available at
\verb{https://archimer.ifremer.fr/doc/00187/29825/} online.
\item User's Manual (ar-um-02-01) 13 July 2010, available at
\verb{http://www.argodatamgt.org/content/download/4729/34634/file/argo-dm-user-manual-version-2.3.pdf},
which is the main document describing argo data.
}
}
\seealso{
The documentation for the \linkS4class{argo} class explains the structure of argo
objects, and also outlines the other functions dealing with them.

Other things related to argo data: 
\code{\link{[[,argo-method}},
\code{\link{[[<-,argo-method}},
\code{\link{argo-class}},
\code{\link{argoGrid}()},
\code{\link{argoNames2oceNames}()},
\code{\link{argo}},
\code{\link{as.argo}()},
\code{\link{handleFlags,argo-method}},
\code{\link{plot,argo-method}},
\code{\link{read.argo.copernicus}()},
\code{\link{subset,argo-method}},
\code{\link{summary,argo-method}}
}
\author{
Dan Kelley
}
\concept{things related to argo data}
