% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_download.R
\name{get_download}
\alias{get_download}
\title{Function to return full download records using \code{site}s, \code{dataset}s, or dataset IDs.}
\usage{
get_download(x, verbose = TRUE)
}
\arguments{
\item{x}{A single numeric dataset ID or a vector of numeric dataset IDs as returned by \code{get_datasets}, or a \code{site}, \code{dataset}, or \code{dataset_list}.}

\item{verbose}{logical; should messages on API call be printed?}
}
\value{
This command returns either object of class \code{"try-error"}' (see \code{\link{try}}) defined by the error returned from the Neotoma API call, or an object of class \code{download_list}, containing a set of \code{download} objects, each with relevant assemblage information and metadata:
The \code{download} object is a list of lists and data frames that describe an assemblage, the constituent taxa, the chronology, site and PIs who contributed the data. The following are important components:

 \item{ \code{dataset} }{A table describing the collection, including dataset information, PI data compatible with \code{\link{get_contact}} and site data compatable with \code{\link{get_site}}.}
 \item{ \code{sample.meta} }{Dataset information for the core, primarily the age-depth model and chronology.  In cases where multiple age models exist for a single record the most recent chronology is provided here.}
 \item{ \code{taxon.list} }{The list of taxa contained within the dataset, unordered, including information that can be used in \code{\link{get_taxa}}}
 \item{ \code{counts} }{The assemblage data for the dataset, arranged with each successive depth in rows and the taxa as columns.  All taxa are described in \code{taxon.list}, the chronology is in \code{sample.data}}
 \item{ \code{lab.data} }{A data frame of laboratory data, such as exotic pollen spike, amount of sample counted, charcoal counts, etc.}
 \item{ \code{chronologies} }{A list of existing chronologies.  If only a single chronology exists for a record then this is the same as the age-model in \code{sample.meta}.}
}
\description{
Using the dataset ID, site object or dataset object, return all records associated with the data as a \code{download_list}.
}
\section{Note}{

The function returns a warning in cases where single taxa are defined by multiple taphonomic characteristics, for example grains that are identified separately as crumpled and torn in the same sample and sums these values within a sample.
In the case that a geochronology dataset is passed to \code{get_download} the function returns a message and a NULL object (that is later excised).  Use \code{get_geochron} for these objects.
The chronologies can be augmented using the function \code{get_chroncontrol}, where the individual chronology objects in \code{chronologies} will consist of a table equivalent to \code{sample.meta} and a \code{chroncontrol} object.
}

\examples{
\dontrun{
#  Search for sites with "Pseudotsuga" pollen that are older than 8kyr BP and
#  that are roughly within western British Columbia:
t8kyr.datasets <- get_dataset(taxonname='*Picea*', loc=c(-90, 41, -89, 44), 
                              ageold = 20000, ageyoung=10000)

#  Returns 20 records (as of 04/04/2013), get the dataset for all records:
pollen.records <- get_download(t8kyr.datasets)

#  Standardize the taxonomies for the different records using the WS64 taxonomy.
compiled.sites <- compile_taxa(pollen.records, list.name='WS64')

#  Extract the Pseudotsuga curves for the sites:
get.curve <- function(x, taxa) {
               if (taxa \%in\% colnames(x$counts)) {
                 count <- x$counts[,taxa]/rowSums(x$counts, na.rm=TRUE)
               } else {
                 count <- rep(0, nrow(x$count))
               }
               data.frame(site = x$dataset$site.data$site.name,
               age = x$sample.meta$age,
               count = count)
             }

curves <- do.call(rbind.data.frame,
                  lapply(compiled.sites, get.curve, taxa = 'Larix/Pseudotsuga'))

#  For illustration, remove the sites with no Pseudotsuga occurance:
curves <- curves[curves$count > 0, ]

smooth.curve <- predict(loess(sqrt(count)~age, data=curves),
                        data.frame(age=seq(20000, 0, by = -100)))
                        
plot(sqrt(count) ~ age, data = curves,
     ylab = '\% Pseudotsuga/Larix', xlab='Calibrated Years BP', pch=19,
     col=rgb(0.1, 0.1, 0.1, 0.1), xlim=c(0, 20000))
lines(seq(20000, 0, by = -100), smooth.curve, lwd=2, lty=2, col=2)

#  This figure shows us an apparent peak in Larix/Pseudotsuga pollen in the 
#  early-Holocene that lends support to a warmer, drier early-Holocene in
#  western North America.
}

}
\references{
Neotoma Project Website: http://www.neotomadb.org
API Reference:  http://api.neotomadb.org/doc/resources/contacts
}
\author{
Simon J. Goring \email{simon.j.goring@gmail.com}
}
\keyword{IO}
\keyword{connection}
