/*
 *  X.509 internal, common functions for writing
 *
 *  Copyright The Mbed TLS Contributors
 *  SPDX-License-Identifier: Apache-2.0
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"); you may
 *  not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *  WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
#include "common.h"
#if defined(MBEDTLS_X509_CSR_WRITE_C) || defined(MBEDTLS_X509_CRT_WRITE_C)

#include "mbedtls/x509_crt.h"
#include "mbedtls/asn1write.h"
#include "mbedtls/error.h"
#include "mbedtls/oid.h"
#include "mbedtls/platform.h"
#include "mbedtls/platform_util.h"
#include "mbedtls/md.h"

#include <string.h>
#include <stdint.h>

#if defined(MBEDTLS_PEM_WRITE_C)
#include "mbedtls/pem.h"
#endif /* MBEDTLS_PEM_WRITE_C */

#if defined(MBEDTLS_USE_PSA_CRYPTO)
#include "psa/crypto.h"
#include "mbedtls/psa_util.h"
#include "md_psa.h"
#endif /* MBEDTLS_USE_PSA_CRYPTO */

#define CHECK_OVERFLOW_ADD(a, b) \
    do                         \
    {                           \
        if (a > SIZE_MAX - (b)) \
        { \
            return MBEDTLS_ERR_X509_BAD_INPUT_DATA; \
        }                            \
        a += b; \
    } while (0)

int mbedtls_x509_write_set_san_common(mbedtls_asn1_named_data **extensions,
                                      const mbedtls_x509_san_list *san_list)
{
    int ret = 0;
    const mbedtls_x509_san_list *cur;
    unsigned char *buf;
    unsigned char *p;
    size_t len;
    size_t buflen = 0;

    for (cur = san_list; cur != NULL; cur = cur->next) {
        switch (cur->node.type) {
            case MBEDTLS_X509_SAN_DNS_NAME:
            case MBEDTLS_X509_SAN_UNIFORM_RESOURCE_IDENTIFIER:
            case MBEDTLS_X509_SAN_IP_ADDRESS:
            case MBEDTLS_X509_SAN_RFC822_NAME:
                CHECK_OVERFLOW_ADD(buflen, cur->node.san.unstructured_name.len);
                CHECK_OVERFLOW_ADD(buflen, 4 + 1);
                break;
            case MBEDTLS_X509_SAN_DIRECTORY_NAME:
            {
                const mbedtls_asn1_named_data *chunk = &cur->node.san.directory_name;
                while (chunk != NULL) {
                    CHECK_OVERFLOW_ADD(buflen, 4 + 1 + 4 + 1);
                    CHECK_OVERFLOW_ADD(buflen, chunk->oid.len);
                    CHECK_OVERFLOW_ADD(buflen, chunk->val.len);
                    chunk = chunk->next;
                }
                CHECK_OVERFLOW_ADD(buflen, 4 + 1);
                break;
            }
            default:
                return MBEDTLS_ERR_X509_FEATURE_UNAVAILABLE;
        }
    }

    CHECK_OVERFLOW_ADD(buflen, 4 + 1);

    buf = mbedtls_calloc(1, buflen);
    if (buf == NULL) {
        return MBEDTLS_ERR_ASN1_ALLOC_FAILED;
    }
    p = buf + buflen;

    cur = san_list;
    len = 0;
    while (cur != NULL) {
        size_t single_san_len = 0;
        switch (cur->node.type) {
            case MBEDTLS_X509_SAN_DNS_NAME:
            case MBEDTLS_X509_SAN_RFC822_NAME:
            case MBEDTLS_X509_SAN_UNIFORM_RESOURCE_IDENTIFIER:
            case MBEDTLS_X509_SAN_IP_ADDRESS:
            {
                const unsigned char *unstructured_name =
                    (const unsigned char *) cur->node.san.unstructured_name.p;
                size_t unstructured_name_len = cur->node.san.unstructured_name.len;

                MBEDTLS_ASN1_CHK_CLEANUP_ADD(single_san_len,
                                             mbedtls_asn1_write_raw_buffer(
                                                 &p, buf,
                                                 unstructured_name, unstructured_name_len));
                MBEDTLS_ASN1_CHK_CLEANUP_ADD(single_san_len, mbedtls_asn1_write_len(
                                                 &p, buf, unstructured_name_len));
                MBEDTLS_ASN1_CHK_CLEANUP_ADD(single_san_len,
                                             mbedtls_asn1_write_tag(
                                                 &p, buf,
                                                 MBEDTLS_ASN1_CONTEXT_SPECIFIC | cur->node.type));
            }
            break;
            case MBEDTLS_X509_SAN_DIRECTORY_NAME:
                MBEDTLS_ASN1_CHK_CLEANUP_ADD(single_san_len,
                                             mbedtls_x509_write_names(&p, buf,
                                                                      (mbedtls_asn1_named_data *) &
                                                                      cur->node
                                                                      .san.directory_name));
                MBEDTLS_ASN1_CHK_CLEANUP_ADD(single_san_len,
                                             mbedtls_asn1_write_len(&p, buf, single_san_len));
                MBEDTLS_ASN1_CHK_CLEANUP_ADD(single_san_len,
                                             mbedtls_asn1_write_tag(&p, buf,
                                                                    MBEDTLS_ASN1_CONTEXT_SPECIFIC |
                                                                    MBEDTLS_ASN1_CONSTRUCTED |
                                                                    MBEDTLS_X509_SAN_DIRECTORY_NAME));
                break;
            default:
                ret = MBEDTLS_ERR_X509_FEATURE_UNAVAILABLE;
                goto cleanup;
        }
        cur = cur->next;
        if (len > SIZE_MAX - single_san_len) {
            ret = MBEDTLS_ERR_X509_BAD_INPUT_DATA;
            goto cleanup;
        }
        len += single_san_len;
    }

    MBEDTLS_ASN1_CHK_CLEANUP_ADD(len, mbedtls_asn1_write_len(&p, buf, len));
    MBEDTLS_ASN1_CHK_CLEANUP_ADD(len,
                                 mbedtls_asn1_write_tag(&p, buf,
                                                        MBEDTLS_ASN1_CONSTRUCTED |
                                                        MBEDTLS_ASN1_SEQUENCE));

    ret = mbedtls_x509_set_extension(extensions,
                                     MBEDTLS_OID_SUBJECT_ALT_NAME,
                                     MBEDTLS_OID_SIZE(MBEDTLS_OID_SUBJECT_ALT_NAME),
                                     0,
                                     buf + buflen - len, len);

    if (p < buf) {
        ret = MBEDTLS_ERR_ASN1_LENGTH_MISMATCH;
    }
cleanup:
    mbedtls_free(buf);
    return ret;
}

#endif /* MBEDTLS_X509_CSR_WRITE_C || MBEDTLS_X509_CRT_WRITE_C */
