% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OptimizerSuccessiveHalving.R
\name{mlr_optimizers_successive_halving}
\alias{mlr_optimizers_successive_halving}
\alias{OptimizerSuccessiveHalving}
\title{Hyperparameter Optimization with Successive Halving}
\source{
Jamieson K, Talwalkar A (2016).
\dQuote{Non-stochastic Best Arm Identification and Hyperparameter Optimization.}
In Gretton A, Robert CC (eds.), \emph{Proceedings of the 19th International Conference on Artificial Intelligence and Statistics}, volume 51 series Proceedings of Machine Learning Research, 240-248.
\url{http://proceedings.mlr.press/v51/jamieson16.html}.
}
\description{
\code{OptimizerSuccessiveHalving} class that implements the successive halving
algorithm. The algorithm samples \code{n} points and evaluates them with the
smallest budget (lower bound of the \code{budget} parameter). With every stage the
budget is increased by a factor of \code{eta} and only the best \code{1/eta} points are
promoted to the next stage. The optimization terminates when the maximum
budget is reached (upper bound of the \code{budget} parameter).

To identify the budget, the user has to specify explicitly which parameter of
the objective function influences the budget by tagging a single parameter in
the search_space (\link[paradox:ParamSet]{paradox::ParamSet}) with \code{"budget"}.
}
\section{Parameters}{

\describe{
\item{\code{n}}{\code{integer(1)}\cr
Number of points in first stage.}
\item{\code{eta}}{\code{numeric(1)}\cr
With every stage, the point budget is increased by a factor of \code{eta}
and only the best \code{1/eta} points are used for the next stage.
Non-integer values are supported, but \code{eta} is not allowed to be less or
equal 1.}
\item{\code{sampler}}{\link[paradox:Sampler]{paradox::Sampler}\cr
Object defining how the samples of the parameter space should be drawn during
the initialization of each bracket. The default is uniform sampling.}
}
}

\section{Archive}{

The \link[bbotk:Archive]{bbotk::Archive} holds the following additional column that is specific
to the successive halving algorithm:
\itemize{
\item \code{stage} (\verb{integer(1))}\cr
Stage index. Starts counting at 0.
}
}

\section{Custom sampler}{

Hyperband supports custom \link[paradox:Sampler]{paradox::Sampler} object for initial
configurations in each bracket.
A custom sampler may look like this (the full example is given in the
\emph{examples} section):\preformatted{# - beta distribution with alpha = 2 and beta = 5
# - categorical distribution with custom probabilities
sampler = SamplerJointIndep$new(list(
  Sampler1DRfun$new(params[[2]], function(n) rbeta(n, 2, 5)),
  Sampler1DCateg$new(params[[3]], prob = c(0.2, 0.3, 0.5))
))
}
}

\section{Runtime}{

The calculation of each bracket currently assumes a linear runtime in the
chosen budget parameter is always given. Hyperband is designed so each
bracket requires approximately the same runtime as the sum of the budget
over all configurations in each bracket is roughly the same. This will not
hold true once the scaling in the budget parameter is not linear
anymore, even though the sum of the budgets in each bracket remains the
same. A possible adaption would be to introduce a trafo, like it is shown in
the \emph{examples} section.
}

\section{Progress Bars}{

\verb{$optimize()} supports progress bars via the package \CRANpkg{progressr}
combined with a \link{Terminator}. Simply wrap the function in
\code{progressr::with_progress()} to enable them. We recommend to use package
\CRANpkg{progress} as backend; enable with \code{progressr::handlers("progress")}.
}

\section{Parallelization}{

In order to support general termination criteria and parallelization, we
evaluate points in a batch-fashion of size \code{batch_size}. The points of one
stage in a bracket are evaluated in one batch. Parallelization is supported
via package \CRANpkg{future} (see \code{\link[mlr3:benchmark]{mlr3::benchmark()}}'s section on
parallelization for more details).
}

\section{Logging}{

Hyperband uses a logger (as implemented in \CRANpkg{lgr}) from package
\CRANpkg{bbotk}.
Use \code{lgr::get_logger("bbotk")} to access and control the logger.
}

\examples{
library(bbotk)
library(data.table)

search_space = domain = ps(
  x1 = p_dbl(-5, 10),
  x2 = p_dbl(0, 15),
  fidelity = p_dbl(1e-2, 1, tags = "budget")
)

# modified branin function
objective = ObjectiveRFunDt$new(
  fun = function(xdt) {
    a = 1
    b = 5.1 / (4 * (pi ^ 2))
    c = 5 / pi
    r = 6
    s = 10
    t = 1 / (8 * pi)
    data.table(y =
      (a * ((xdt[["x2"]] -
      b * (xdt[["x1"]] ^ 2L) +
      c * xdt[["x1"]] - r) ^ 2) +
      ((s * (1 - t)) * cos(xdt[["x1"]])) +
      s - (5 * xdt[["fidelity"]] * xdt[["x1"]])))
  },
  domain = domain,
  codomain = ps(y = p_dbl(tags = "minimize"))
)

instance = OptimInstanceSingleCrit$new(
  objective = objective,
  search_space = search_space,
  terminator = trm("none")
)

optimizer = opt("successive_halving")

# modifies the instance by reference
optimizer$optimize(instance)

# best scoring evaluation
instance$result

# all evaluations
as.data.table(instance$archive)
}
\section{Super class}{
\code{\link[bbotk:Optimizer]{bbotk::Optimizer}} -> \code{OptimizerSuccessiveHalving}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{OptimizerSuccessiveHalving$new()}}
\item \href{#method-clone}{\code{OptimizerSuccessiveHalving$clone()}}
}
}
\if{html}{
\out{<details open ><summary>Inherited methods</summary>}
\itemize{
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="format">}\href{../../bbotk/html/Optimizer.html#method-format}{\code{bbotk::Optimizer$format()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="optimize">}\href{../../bbotk/html/Optimizer.html#method-optimize}{\code{bbotk::Optimizer$optimize()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="print">}\href{../../bbotk/html/Optimizer.html#method-print}{\code{bbotk::Optimizer$print()}}\out{</span>}
}
\out{</details>}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerSuccessiveHalving$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerSuccessiveHalving$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
