% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/benchmark.R
\name{benchmark}
\alias{benchmark}
\title{Benchmark Multiple Learners on Multiple Tasks}
\usage{
benchmark(design, ctrl = list())
}
\arguments{
\item{design}{:: \code{\link[=data.frame]{data.frame()}}\cr
Data frame (or \code{\link[=data.table]{data.table()}}) with three columns: "task", "learner", and "resampling".
Each row defines a resampling by providing a \link{Task}, \link{Learner} and a \link{Resampling} strategy.
All resamplings must be properly instantiated.
The helper function \code{\link[=expand_grid]{expand_grid()}} can assist in generating an exhaustive design (see examples) and
instantiate the \link{Resampling}s per \link{Task}.}

\item{ctrl}{:: (named \code{list()})\cr
Object to control learner execution. See \code{\link[=mlr_control]{mlr_control()}} for details.
Note that per default, fitted learner models are discarded after the prediction in order to save
some memory.}
}
\value{
\link{BenchmarkResult}.
}
\description{
Runs a benchmark on arbitrary combinations of learners, tasks, and resampling strategies (possibly in parallel).
Resamplings which are not already instantiated will be instantiated automatically.
However, these auto-instantiated resamplings will not be synchronized per task, i.e. different learners will
work on different splits of the same task.

To generate exhaustive designs and automatically instantiate resampling strategies per task, use \code{\link[=expand_grid]{expand_grid()}}.
}
\note{
The fitted models are discarded after the predictions have been scored in order to reduce memory consumption.
If you need access to the models for later analysis, set \code{store_models} to \code{TRUE} via \code{\link[=mlr_control]{mlr_control()}}.
}
\section{Parallelization}{


This function can be parallelized with the \CRANpkg{future} package.
One job is one resampling iteration, and all jobs are send to an apply function
from \CRANpkg{future.apply} in a single batch.
To select a parallel backend, use \code{\link[future:plan]{future::plan()}}.
}

\section{Syntactic Sugar}{


The mlr3 package provides some shortcuts to ease the creation of its objects.

First, instead of an object, it is possible to pass a string identifier which is used to lookup the object in a \link[mlr3misc:Dictionary]{mlr3misc::Dictionary}:
\itemize{
\item \link{Task} in \link{mlr_tasks}.
\item \link{Learner} in \link{mlr_learners}.
\item \link{Resampling} in \link{mlr_resamplings}.
\item \link{Measure} in \link{mlr_measures}.
}

Additionally, each task type has an associated default measure (stored in \link{mlr_reflections}) which is used as a fallback if no other measure is provided.
Classification tasks default to the classification error in \link[=mlr_measures_classif.ce]{"classif.ce"}, regression tasks to the mean squared error in \link[=mlr_measures_regr.mse]{"regr.mse"}.
}

\examples{
# benchmarking with expand_grid()
tasks = mlr_tasks$mget(c("iris", "sonar"))
learners = mlr_learners$mget(c("classif.featureless", "classif.rpart"))
resamplings = mlr_resamplings$mget("cv3")

design = expand_grid(tasks, learners, resamplings)
print(design)

set.seed(123)
bmr = benchmark(design)

## data of all resamplings
head(as.data.table(bmr))

## aggregated performance values
aggr = bmr$aggregate()
print(aggr)

## Extract predictions of first resampling result
rr = aggr$resample_result[[1]]
as.data.table(rr$prediction)

# benchmarking with a custom design:
# - fit classif.featureless on iris with a 3-fold CV
# - fit classif.rpart on sonar using a holdout
design = data.table::data.table(
  task = mlr_tasks$mget(c("iris", "sonar")),
  learner = mlr_learners$mget(c("classif.featureless", "classif.rpart")),
  resampling = mlr_resamplings$mget(c("cv3", "holdout"))
)

## instantiate resamplings
design$resampling = Map(
  function(task, resampling) resampling$clone()$instantiate(task),
  task = design$task, resampling = design$resampling
)

## calculate benchmark
bmr = benchmark(design)
print(bmr)

## get the training set of the 2nd iteration of the featureless learner on iris
rr = bmr$aggregate()[learner_id == "classif.featureless"]$resample_result[[1]]
rr$resampling$train_set(2)
}
