% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crate.R
\name{crate}
\alias{crate}
\title{Crate a function to share with another process}
\usage{
crate(.fn, ...)
}
\arguments{
\item{.fn}{A fresh formula or function. "Fresh" here means that
they should be declared in the call to `crate()`. See examples if
you need to crate a function that is already defined. Formulas
are converted to purrr-like lambda functions using
[rlang::as_function()].}

\item{...}{Arguments to declare in the environment of `.fn`. If a
name is supplied, the object is assigned to that name. Otherwise
the argument is automatically named after itself.}
}
\description{
`crate()` creates functions in a self-contained environment
(technically, a child of the base environment). This has two
advantages:

* They can easily be executed in another process.

* Their effects are reproducible. You can run them locally with the
  same results as on a different process.

Creating self-contained functions requires some care, see section
below.
}
\section{Creating self-contained functions}{


* They should call package functions with an explicit `::`
  namespace. This includes packages in the default search path with
  the exception of the base package. For instance `var()` from the
  stats package must be called with its namespace prefix:
  `stats::var(x)`.

* They should declare any data they depend on. You can declare data
  by supplying additional arguments or by unquoting objects with `!!`.
}

\examples{
# You can create functions using the ordinary notation:
crate(function(x) stats::var(x))

# Or the formula notation:
crate(~stats::var(.x))

# Declare data by supplying named arguments. You can test you have
# declared all necessary data by calling your crated function:
na_rm <- TRUE
fn <- crate(~stats::var(.x, na.rm = na_rm))
try(fn(1:10))

# Arguments are automatically named after themselves so that the
# following are equivalent:
crate(~stats::var(.x, na.rm = na_rm), na_rm = na_rm)
crate(~stats::var(.x, na.rm = na_rm), na_rm)

# However if you supply a complex expression, do supply a name!
crate(~stats::var(.x, na.rm = na_rm), !na_rm)
crate(~stats::var(.x, na.rm = na_rm), na_rm = na_rm)

# For small data it is handy to unquote instead. Unquoting inlines
# objects inside the function. This is less verbose if your
# function depends on many small objects:
fn <- crate(~stats::var(.x, na.rm = !!na_rm))
fn(1:10)

# One downside is that the individual sizes of unquoted objects
# won't be shown in the crate printout:
fn


# The function or formula you pass to crate() should defined inside
# the crate() call, i.e. you can't pass an already defined
# function:
fn <- function(x) toupper(x)
try(crate(fn))

# If you really need to crate an existing function, you can
# explicitly set its environment to the crate environment with the
# set_env() function from rlang:
crate(rlang::set_env(fn))
}
