% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_arrow.R
\docType{data}
\name{geom_arrow}
\alias{geom_arrow}
\alias{geom_vector}
\alias{GeomArrow}
\alias{StatArrow}
\title{Arrows}
\usage{
geom_arrow(mapping = NULL, data = NULL, stat = "arrow",
  position = "identity", ..., start = 0, direction = c("ccw", "cw"),
  pivot = 0.5, preserve.dir = TRUE, min.mag = 0, skip = 0,
  skip.x = skip, skip.y = skip, arrow.angle = 15,
  arrow.length = 0.5, arrow.ends = "last", arrow.type = "closed",
  arrow = grid::arrow(arrow.angle, unit(arrow.length, "lines"), ends =
  arrow.ends, type = arrow.type), lineend = "butt", na.rm = FALSE,
  show.legend = NA, inherit.aes = TRUE)

geom_vector(mapping = NULL, data = NULL, stat = "arrow",
  position = "identity", ..., start = 0, direction = c("ccw", "cw"),
  pivot = 0.5, preserve.dir = FALSE, min.mag = 0, skip = 0,
  skip.x = skip, skip.y = skip, arrow.angle = 15,
  arrow.length = 0.5, arrow.ends = "last", arrow.type = "closed",
  arrow = grid::arrow(arrow.angle, unit(arrow.length, "lines"), ends =
  arrow.ends, type = arrow.type), lineend = "butt", na.rm = FALSE,
  show.legend = NA, inherit.aes = TRUE)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[=aes]{aes()}} or
\code{\link[=aes_]{aes_()}}. If specified and \code{inherit.aes = TRUE} (the
default), it is combined with the default mapping at the top level of the
plot. You must supply \code{mapping} if there is no plot mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[=ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[=fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data.}

\item{stat}{The statistical transformation to use on the data for this
layer, as a string.}

\item{position}{Position adjustment, either as a string, or the result of
a call to a position adjustment function.}

\item{...}{Other arguments passed on to \code{\link[=layer]{layer()}}. These are
often aesthetics, used to set an aesthetic to a fixed value, like
\code{colour = "red"} or \code{size = 3}. They may also be parameters
to the paired geom/stat.}

\item{start}{starting angle for rotation in degrees}

\item{direction}{direction of rotation (counter-clockwise or clockwise)}

\item{pivot}{numeric indicating where to pivot the arrow where 0 means at the
beginning and 1 means at the end.}

\item{preserve.dir}{logical indicating whether to preserve direction or not}

\item{min.mag}{minimum magnitude for plotting vectors}

\item{skip, skip.x, skip.y}{numeric specifying number of gridpoints not to draw
in the x and y direction}

\item{arrow.length, arrow.angle, arrow.ends, arrow.type}{parameters passed to
\link[grid:arrow]{grid::arrow}}

\item{arrow}{specification for arrow heads, as created by arrow().}

\item{lineend}{Line end style (round, butt, square).}

\item{na.rm}{If \code{FALSE}, the default, missing values are removed with
a warning. If \code{TRUE}, missing values are silently removed.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[=borders]{borders()}}.}
}
\description{
Parametrization of \link[ggplot2:geom_segment]{ggplot2::geom_segment} either by location and displacement
or by magnitude and angle with default arrows. \code{geom_arrow()} is the same as
\code{geom_vector()} but defaults to preserving the direction under coordinate
transformation and different plot ratios.
}
\details{
Direction and start allows to work with different standards. For the
meteorological standard, for example, use \code{star = -90} and \code{direction = "cw"}.
}
\section{Aesthetics}{

\code{geom_vector} understands the following aesthetics (required aesthetics are in bold)

\itemize{
\item \strong{x}
\item \strong{y}
\item either \strong{mag} and \strong{angle}, or \strong{dx} and \strong{dy}
\item \code{alpha}
\item \code{colour}
\item \code{linetype}
\item \code{size}
\item \code{lineend}
}
}

\examples{
library(data.table)
library(ggplot2)

data(seals)
# If the velocity components are in the same units as the axis,
# geom_vector() (or geom_arrow(preserve.dir = TRUE)) might be a better option
ggplot(seals, aes(long, lat)) +
    geom_arrow(aes(dx = delta_long, dy = delta_lat), skip = 1, color = "red") +
    geom_vector(aes(dx = delta_long, dy = delta_lat), skip = 1) +
    scale_mag()

data(geopotential)
geopotential <- copy(geopotential)[date == date[1]]
geopotential[, gh.z := Anomaly(gh), by = .(lat)]
geopotential[, c("u", "v") := GeostrophicWind(gh.z, lon, lat)]

(g <- ggplot(geopotential, aes(lon, lat)) +
    geom_arrow(aes(dx = dlon(u, lat), dy = dlat(v)), skip.x = 3, skip.y = 2,
               color = "red") +
    geom_vector(aes(dx = dlon(u, lat), dy = dlat(v)), skip.x = 3, skip.y = 2) +
    scale_mag(max_size = 2, guide = "none"))

# A dramatic illustration of the difference between arrow and vector
g + coord_polar()

# When plotting winds in a lat-lon grid, a good way to have both
# the correct direction and an interpretable magnitude is to define
# the angle by the longitud and latitude displacement and the magnitude
# by the wind velocity. That way arrows are always parallel to streamlines
# and their magnitude are in the correct units.
ggplot(geopotential, aes(lon, lat)) +
    geom_contour(aes(z = gh.z)) +
    geom_vector(aes(angle = atan2(dlat(v), dlon(u, lat))*180/pi,
                   mag = Mag(v, u)), skip = 1, pivot = 0.5) +
    scale_mag()

# Sverdrup transport
library(data.table)
b <- 10
d <- 10
grid <- as.data.table(expand.grid(x = seq(1, d, by = 0.5),
                                  y = seq(1, b, by = 0.5)))
grid[, My := -sin(pi*y/b)*pi/b]
grid[, Mx := -pi^2/b^2*cos(pi*y/b)*(d - x)]

ggplot(grid, aes(x, y)) +
    geom_arrow(aes(dx = Mx, dy = My))

}
\seealso{
Other ggplot2 helpers: \code{\link{DivideTimeseries}},
  \code{\link{MakeBreaks}}, \code{\link{WrapCircular}},
  \code{\link{geom_contour2}},
  \code{\link{geom_contour_fill}},
  \code{\link{geom_label_contour}},
  \code{\link{geom_relief}}, \code{\link{geom_streamline}},
  \code{\link{guide_colourstrip}},
  \code{\link{map_labels}}, \code{\link{reverselog_trans}},
  \code{\link{scale_divergent}},
  \code{\link{scale_longitude}}, \code{\link{stat_na}},
  \code{\link{stat_subset}}
}
\concept{ggplot2 helpers}
\keyword{datasets}
