\name{lsei}
\alias{lsei}
\title{Least Squares with Equalities and Inequalities}
\description{Solves an lsei inverse problem (Least Squares with Equality and Inequality constraints)
 \deqn{\min(||Ax-b||^2)} subject to        
 \deqn{Ex=f}
 \deqn{Gx>=h}
 
 uses either subroutine lsei (FORTRAN) from the LINPACK package, or \code{\link[quadprog]{solve.QP}} from R-package quadprog
 \cr In case the equality constraints \eqn{Ex=f} cannot be satisfied, a generalized inverse solution residual vector length is obtained for \eqn{f-Ex}.
 \cr This is the minimal length possible for \eqn{||f-Ex||^2}.}
\usage{lsei(A=NULL, B=NULL, E=NULL, F=NULL, G=NULL, H=NULL, 
Wx=NULL, Wa=NULL, type=1, tol=sqrt(.Machine$double.eps),
tolrank=NULL, fulloutput=FALSE, verbose=TRUE)}
\arguments{
  \item{A }{numeric matrix containing the coefficients of the quadratic function to be minimised, \eqn{||Ax-B||}; if the columns of A have a names attribute, they will be used to label the output}
  \item{B }{numeric vector containing the right-hand side of the quadratic function to be minimised}
  \item{E }{numeric matrix containing the coefficients of the equality constraints, \eqn{Ex=F}; if the columns of E have a names attribute, and the columns of A do not, they will be used to label the output}
  \item{F }{numeric vector containing the right-hand side of the equality constraints}
  \item{G }{numeric matrix containing the coefficients of the inequality constraints, \eqn{Gx>=H}; if the columns of G have a names attribute, and the columns of A and E do not, they will be used to label the output}
  \item{H }{numeric vector containing the right-hand side of the inequality constraints}
  \item{Wx }{numeric vector with weighting coefficients of unknowns (length = number of unknowns)}
  \item{Wa }{numeric vector with weighting coefficients of the quadratic function (Ax-B) to be minimised (length = number of number of rows of A)}
  \item{type }{integer code determining algorithm to use 1=\code{lsei} , 2=\code{solve.QP} from R-package \code{quadprog} (see note)}
  \item{tol }{tolerance (for singular value decomposition, equality and inequality constraints)}
  \item{tolrank }{Only if \code{type} = 1; if not NULL then \code{tolrank} should be a two-valued vector containing the rank determination tolerance for the equality constraint equations from
(1st value) and for the reduced least squares equations (2nd value)}
  \item{fulloutput }{if TRUE, also returns the covariance matrix of the solution and the rank of the equality constraints - only if \code{type} = 1}
  \item{verbose }{logical to print error messages}
}
\value{
  a list containing:
  \item{X }{vector containing the solution of the least squares problem.}
  \item{residualNorm }{scalar, the sum of absolute values of residuals of equalities and violated inequalities}
  \item{solutionNorm }{scalar, the value of the minimised quadratic function at the solution, i.e. the value of \eqn{||Ax-b||^2}}
  \item{IsError }{logical, TRUE if an error occurred}
  \item{type }{the string "lsei", such that how the solution was obtained can be traced}
  \item{covar }{covariance matrix of the solution; only returned if \code{fulloutput} = TRUE}
  \item{RankEq }{rank of the equality constraint matrix.; only returned if \code{fulloutput} = TRUE}
  \item{RankApp }{rank of the reduced least squares problem (approximate equations); only returned if \code{fulloutput} = TRUE}
}
\author{Karline Soetaert <k.soetaert@nioo.knaw.nl>}
\examples{x <- 1:5
y <- c(9,8,6,7,5)
plot(x,y,main="Polynomial fitting, using lsei",cex=1.5,pch=16,
     ylim=c(4,10))
A <- cbind(rep(1,5),x)
B <- y
cf <- lsei(A,B)$X
abline(coef=cf)
A <- cbind(A,x^2)
cf <- lsei(A,B)$X
curve(cf[1]+cf[2]*x+cf[3]*x^2,add=TRUE,lty=2)
A <- cbind(A,x^3)
cf <- lsei(A,B)$X
curve(cf[1]+cf[2]*x+cf[3]*x^2+cf[4]*x^3,add=TRUE,lty=3)
A <- cbind(A,x^4)
cf <- lsei(A,B)$X
curve(cf[1]+cf[2]*x+cf[3]*x^2+cf[4]*x^3+cf[5]*x^4,add=TRUE,lty=4)
legend("bottomleft",c("1st-order","2nd-order","3rd-order","4th-order"),
       lty=1:4)}
\references{K. H. Haskell and R. J. Hanson, An algorithm for linear least squares problems with equality and
nonnegativity constraints, Report SAND77-0552, Sandia Laboratories, June 1978.
\cr K. H. Haskell and R. J. Hanson, Selected algorithms for the linearly constrained least squares problem - a
users guide, Report SAND78-1290, Sandia Laboratories,August 1979.
\cr K. H. Haskell and R. J. Hanson, An algorithm for linear least squares problems with equality and
nonnegativity constraints, Mathematical Programming 21 (1981), pp. 98-118.
\cr R. J. Hanson and K. H. Haskell, Two algorithms for the linearly constrained least squares problem, ACM
Transactions on Mathematical Software, September 1982.
\cr  Berwin A. Turlach R and Andreas Weingessel (2007). quadprog: Functions to solve
  Quadratic Programming Problems.. R package version 1.4-11. S original by Berwin A. Turlach R port by Andreas Weingessel}
\seealso{\code{\link{ldei}}, \code{\link{linp}}, 
\cr\code{\link[quadprog]{solve.QR}} the original function from package quadprog}
\note{See comments in the original code for more details; these comments are included in the \file{docs} subroutine of the package}
\keyword{algebra}
\keyword{array}
\keyword{optimize}

