% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sur.R
\name{estim.sur}
\alias{estim.sur}
\title{Estimate a SUR Model}
\usage{
estim.sur(
  y,
  x,
  addIntercept = TRUE,
  searchSigMaxIter = 0,
  searchSigMaxProb = 0.1,
  restriction = NULL,
  newX = NULL,
  pcaOptionsY = NULL,
  pcaOptionsX = NULL,
  simFixSize = 0,
  simTrainFixSize = 0,
  simTrainRatio = 0.75,
  simSeed = 0,
  simMaxConditionNumber = Inf,
  printMsg = FALSE
)
}
\arguments{
\item{y}{A matrix of endogenous data with variables in the columns.}

\item{x}{A matrix of exogenous data with variables in the columns.}

\item{addIntercept}{If \code{TRUE}, intercept is added automatically to \code{x}.}

\item{searchSigMaxIter}{An integer for the maximum number of iterations in searching for significant coefficients. Use 0 to disable the search.}

\item{searchSigMaxProb}{A number for the maximum value of type I error to be used in searching for significant coefficients. If p-value is less than this, it is interpreted as significant.}

\item{restriction}{A \code{km x q} matrix of restrictions where \code{m=ncols(y)}, \code{k=ncols(x)} and \code{q} is the number of unrestricted coefficients.}

\item{newX}{A matrix with new exogenous data to be used in the projections. Its number of columns must be equal to \code{x}. It can be \code{NULL}.}

\item{pcaOptionsY}{A list of options to use principal components of the \code{y}, instead of the actual values. Set \code{NULL} to disable. Use \code{\link[=get.options.pca]{get.options.pca()}} for initialization.}

\item{pcaOptionsX}{A list of options to use principal components of the \code{x}, instead of the actual values. Set \code{NULL} to disable. Use \code{\link[=get.options.pca]{get.options.pca()}} for initialization.}

\item{simFixSize}{An integer that determines the number of out-of-sample simulations. Use zero to disable the simulation.}

\item{simTrainFixSize}{An integer representing the number of data points in the training sample in the out-of-sample simulation. If zero, \code{trainRatio} will be used.}

\item{simTrainRatio}{A number representing the size of the training sample relative to the available size, in the out-of-sample simulation. It is effective if \code{trainFixSize} is zero.}

\item{simSeed}{A seed for the random number generator. Use zero for a random value.}

\item{simMaxConditionNumber}{A number for the maximum value for the condition number in the simulation.}

\item{printMsg}{Set to \code{TRUE} to enable printing some details.}
}
\value{
A nested list with the following items:
\item{counts}{Information about different aspects of the estimation such as the number of observation, number of exogenous variables, etc.}
\item{estimations}{Estimated coefficients, standard errors, z-statistics, p-values, etc.}
\item{metrics}{Value of different goodness of fit and out-of-sample performance metrics. }
\item{projections}{Information on the projected values, if \code{newX} is provided.}
\item{info}{Some other general information.}
}
\description{
Use this function to estimate a Seemingly Unrelated Regression model.
}
\details{
Seemingly Unrelated Regression (SUR) is a type of statistical model that includes multiple regression equations.
The general form of an SUR model with m equations can be written as: \eqn{y_i=X_i\beta_i+\epsilon_i}, where $i=1\ldots m$ determines the index of the equation.
In this model, each equation may have different sets of independent variables and it is assumed that the disturbances between the equations are correlated.
The OLS estimator is a consistent estimator for this model, but it is not generally asymptotically efficient (except when disturbances are uncorrelated between equations or each equation contains exactly the same set of regressors).
The OLS variance matrix is used to calculate the Feasible Generalized Least Squares (FGLS) estimator, which is both consistent and asymptotically efficient (under regularity conditions).

In the current implementation, this function focuses on zero restrictions and/or significance search.
Therefore, there is a common \code{x} argument for all equations.
In fact, the main purpose of exporting this method is to show the inner calculations of the search process in \link{search.sur} function.
Note that the focus in \code{ldt} is model uncertainty and for more sophisticated implementations of the FGLS estimator, you may consider using other packages such as \code{systemfit}.
}
\examples{
# Example 1 (simulation, small model):
sample <- sim.sur(sigma = 2L, coef = 3L, nObs = 100)
#    Use systemfit to estimate:
exp_names <- paste0(colnames(sample$x), collapse = " + ")
fmla <- lapply(1:ncol(sample$y), function(i) as.formula(paste0("Y", i, " ~ -1 +", exp_names)))
fit <- systemfit::systemfit(fmla, data = data.frame(sample$y, sample$x), method = "SUR")
#    Use estim.sur function in this package:
fit2 <- estim.sur(sample$y, sample$x, addIntercept = FALSE)
coefs <- fit2$estimations$coefs # coefficient matrix
table <- coefs.table(fit2)


# Example 2 (simulation, large model with significancy search):
num_obs <- 100
num_x_ir <- 20
sample <- sim.sur(sigma = 2L, coef = 3L, nObs = num_obs)
x_ir <- matrix(rnorm(num_obs * num_x_ir), ncol = num_x_ir) # irrelevant sample
data_x <- data.frame(sample$x, x_ir)
colnames(data_x) <- c(colnames(sample$x), paste0("z", 1:num_x_ir))

fit3 <- estim.sur(sample$y, data_x, addIntercept = FALSE,
                  searchSigMaxIter = 100, searchSigMaxProb = 0.05)
coefs <- fit2$estimations$coefs # coefficient matrix, probably with lots of zero restrictions

# Example 3 (simulation, large model with PCA):
# I use the previous example's data
fit3 <- estim.sur(sample$y, data_x, addIntercept = FALSE,
                  pcaOptionsX = get.options.pca(2,4))
coefs <- fit3$estimations$coefs # has intercept and the first exogenous variable and 4 PCs
table <- coefs.table(fit3, expList = 2) # doesn't display PC coefficients
}
\seealso{
\link{search.sur}, \link{search.sur.stepwise}
}
