% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_functions.R
\name{h2o_automl}
\alias{h2o_automl}
\alias{plot.h2o_automl}
\alias{print.h2o_automl}
\title{Automated H2O's AutoML}
\usage{
h2o_automl(
  df,
  y = "tag",
  ignore = NULL,
  train_test = NA,
  split = 0.7,
  weight = NULL,
  target = "auto",
  balance = FALSE,
  impute = FALSE,
  no_outliers = TRUE,
  unique_train = TRUE,
  center = FALSE,
  scale = FALSE,
  thresh = 10,
  seed = 0,
  nfolds = 5,
  max_models = 3,
  max_time = 10 * 60,
  start_clean = FALSE,
  exclude_algos = c("StackedEnsemble", "DeepLearning"),
  include_algos = NULL,
  plots = TRUE,
  alarm = TRUE,
  quiet = FALSE,
  print = TRUE,
  save = FALSE,
  subdir = NA,
  project = "AutoML Results",
  verbosity = NULL,
  ...
)

\method{plot}{h2o_automl}(x, ...)

\method{print}{h2o_automl}(x, importance = TRUE, ...)
}
\arguments{
\item{df}{Dataframe. Dataframe containing all your data, including
the dependent variable labeled as \code{'tag'}. If you want to define
which variable should be used instead, use the \code{y} parameter.}

\item{y}{Variable or Character. Name of the dependent variable or response.}

\item{ignore}{Character vector. Force columns for the model to ignore}

\item{train_test}{Character. If needed, \code{df}'s column name with 'test'
and 'train' values to split}

\item{split}{Numeric. Value between 0 and 1 to split as train/test
datasets. Value is for training set. Set value to 1 to train with all
available data and test with same data (cross-validation will still be
used when training). If \code{train_test} is set, value will be overwritten
with its real split rate.}

\item{weight}{Column with observation weights. Giving some observation a
weight of zero is equivalent to excluding it from the dataset; giving an
observation a relative weight of 2 is equivalent to repeating that
row twice. Negative weights are not allowed.}

\item{target}{Value. Which is your target positive value? If
set to \code{'auto'}, the target with largest \code{mean(score)} will be
selected. Change the value to overwrite. Only used when binary
categorical model.}

\item{balance}{Boolean. Auto-balance train dataset with under-sampling?}

\item{impute}{Boolean. Fill \code{NA} values with MICE?}

\item{no_outliers}{Boolean/Numeric. Remove \code{y}'s outliers from the dataset?
Will remove those values that are farther than n standard deviations from
the dependent variable's mean (Z-score). Set to \code{TRUE} for default (3)
or numeric to set a different multiplier.}

\item{unique_train}{Boolean. Keep only unique row observations for training data?}

\item{center, scale}{Boolean. Using the base function scale, do you wish
to center and/or scale all numerical values?}

\item{thresh}{Integer. Threshold for selecting binary or regression
models: this number is the threshold of unique values we should
have in \code{'tag'} (more than: regression; less than: classification)}

\item{seed}{Integer. Set a seed for reproducibility. AutoML can only
guarantee reproducibility if max_models is used because max_time is
resource limited.}

\item{nfolds}{Number of folds for k-fold cross-validation. Must be >= 2; defaults to 5. Use 0 to disable cross-validation;
this will also disable Stacked Ensemble (thus decreasing the overall model performance).}

\item{max_models, max_time}{Numeric. Max number of models and seconds
you wish for the function to iterate. Note that max_models guarantees
reproducibility and max_time not (because it depends entirely on your
machine's computational characteristics)}

\item{start_clean}{Boolean. Erase everything in the current h2o
instance before we start to train models? You may want to keep other models
or not. To group results into a custom common AutoML project, you may
use \code{project_name} argument.}

\item{exclude_algos, include_algos}{Vector of character strings. Algorithms
to skip or include during the model-building phase. Set NULL to ignore.
When both are defined, only \code{include_algos} will be valid.}

\item{plots}{Boolean. Create plots objects?}

\item{alarm}{Boolean. Ping (sound) when done. Requires \code{beepr}.}

\item{quiet}{Boolean. Keep quiet? If not, informative messages will be shown.}

\item{print}{Boolean. Print summary when process ends?}

\item{save}{Boolean. Do you wish to save/export results into your
working directory?}

\item{subdir}{Character. In which directory do you wish to save
the results? Working directory as default.}

\item{project}{Character. Your project's name}

\item{verbosity}{Verbosity of the backend messages printed during training; Optional.
Must be one of NULL (live log disabled), "debug", "info", "warn", "error". Defaults to "warn".}

\item{...}{Additional parameters on \code{h2o::h2o.automl}}

\item{x}{h2o_automl object}

\item{importance}{Boolean. Print important variables?}
}
\value{
List. Trained model, predicted scores and datasets used, performance
metrics, parameters, importance data.frame, seed, and plots when \code{plots=TRUE}.
}
\description{
This function lets the user create a robust and fast model, using
H2O's AutoML function. The result is a list with the best model,
its parameters, datasets, performance metrics, variables
importance, and plots. Read more about the \code{h2o_automl()} pipeline
\href{https://laresbernardo.github.io/lares/articles/h2o_automl.html}{here}.
}
\section{List of algorithms}{

\href{https://docs.h2o.ai/h2o/latest-stable/h2o-docs/automl.html}{-> Read more here}
\describe{
  \item{DRF}{Distributed Random Forest, including Random Forest (RF)
  and Extremely-Randomized Trees (XRT)}
  \item{GLM}{Generalized Linear Model}
  \item{XGBoost}{eXtreme Grading Boosting}
  \item{GBM}{Gradient Boosting Machine}
  \item{DeepLearning}{Fully-connected multi-layer artificial neural network}
  \item{StackedEnsemble}{Stacked Ensemble}
}
}

\section{Methods}{

\describe{
  \item{print}{Use \code{print} method to print models stats and summary}
  \item{plot}{Use \code{plot} method to plot results using \code{mplot_full()}}
}
}

\examples{
\dontrun{
# CRAN
data(dft) # Titanic dataset
dft <- subset(dft, select = -c(Ticket, PassengerId, Cabin))

# Classification: Binomial - 2 Classes
r <- h2o_automl(dft, y = Survived, max_models = 1, impute = FALSE, target = "TRUE", alarm = FALSE)

# Let's see all the stuff we have inside:
lapply(r, names)

# Classification: Multi-Categorical - 3 Classes
r <- h2o_automl(dft, Pclass, ignore = c("Fare", "Cabin"), max_time = 30, plots = FALSE)

# Regression: Continuous Values
r <- h2o_automl(dft, y = "Fare", ignore = c("Pclass"), exclude_algos = NULL, quiet = TRUE)
print(r)

# WITH PRE-DEFINED TRAIN/TEST DATAFRAMES
splits <- msplit(dft, size = 0.8)
splits$train$split <- "train"
splits$test$split <- "test"
df <- rbind(splits$train, splits$test)
r <- h2o_automl(df, "Survived", max_models = 1, train_test = "split")
}
}
\seealso{
Other Machine Learning: 
\code{\link{ROC}()},
\code{\link{conf_mat}()},
\code{\link{export_results}()},
\code{\link{gain_lift}()},
\code{\link{h2o_predict_MOJO}()},
\code{\link{h2o_selectmodel}()},
\code{\link{impute}()},
\code{\link{iter_seeds}()},
\code{\link{lasso_vars}()},
\code{\link{model_metrics}()},
\code{\link{model_preprocess}()},
\code{\link{msplit}()}
}
\concept{Machine Learning}
