#' cwd_bm_df()
#' 
#' @description
#' cwd_bm_df() function calculates the biomass of Coarse Woody Debris using species-specific national emission factors.
#' It is an internal function used within the cwd_biomass_nfi() function.
#' 
#' @param data : data
#' @noRd

## Calculating carbon storage --------------------------------------------------------------
## species(Decay class) ~ Volume*(Wood density)*(Carbon fraction)*(44/12)-----------------
## 부후등급 및 수종 ~ 추정간재적*(목재기본밀도)*(탄소전환계수)*(44/12)-----------------
cwd_bm_df <- function(data){
  
  
  output <- data  %>% mutate(species_bm = case_when(
    
    ##species-specific
    data$SPCD =="14994" ~ "14994", # Pinus densiflora (소나무)
    data$SPCD =="14964" ~ "14964" , # Larix kaempferi (일본잎갈나무)
    data$SPCD =="14987" ~ "14987", # Pinus rigida (리기다나무)
    data$SPCD =="14973" ~ "14973", # Pinus koraiensis (잣나무)
    data$SPCD =="6617" ~ "6617" , # Quercus variabilis (굴참나무)
    data$SPCD =="6556" ~ "6556" , # Quercus mongolica (신갈나무)
    data$SPCD =="6512" ~ "6512" , # Quercus acutissima (상수리나무)
    
    (data$CONDEC_CLASS_CD ==1) ~ "OTHER_DEC" , ## Other deciduous species
    (data$CONDEC_CLASS_CD ==0) ~ "OTHER_CON",  ## Other conifer species
    TRUE ~ as.character(NA)
    
  ))
  
  
  output <- left_join(output, cwd_bio_coeff, by= c("species_bm" ="SPCD", "DECAYCD" = "decay_class"))
  
  ## Calculating biomass--------------------------------------------------------------
  ## species(Decay class) ~ Volume*(Wood density)*(Carbon fraction)-------------------------------
  output$carbon_stock <- (output$VOL)*(output$wood_density)*(output$cf)
  ## Calculating carbon dioxide storage--------------------------------------------------------------
  ## species(Decay class) ~ Volume*(Wood density)*(Carbon fraction)*(44/12)-------------------------------
  output$co2_stock <- (output$VOL)*(output$wood_density)*(output$cf)*(44/12)
  
  return(output)
  
}







#' Calculate biomass of Coarse Woody Debris
#' 
#' @description
#' cwd_biomass_nfi() function estimates volume, carbon storage and carbon dioxide storage of Coarse Woody Debris (CWD).
#' It can estimate individual plots, the entire study area, or specific groups within the study area using parameters `byplot`, `plotgrp` and `treegrp`.
#' Users can choose the criteria for post-stratification using the `strat` parameter.
#' Users can specify whether to focus only on Stocked land using the `stockedland` parameter.

#' @details
#' This function calculates CWD biomass using methodologies employed for national statistics (mean, variance, standard error, relative standard error):
#' - Applies national carbon emission factors to calculate CWD biomass at the individual tree level.
#' - Estimates biomass per hectare at the plot level.
#' - Uses the Double Sampling for Post-stratification (DSS) method to derive annual statistics.
#' - Applies the Weighted Moving Average (WMA) method to integrate annual statistics from the 20% of plots surveyed each year into a single time point.
#' 
#' @param data : A `list` generated by \code{\link{read_nfi}} that contains 'plot' and 'cwd' data frames.
#' @param byplot :  A logical flag (default FALSE); if TRUE, calculates statistics for each plot separately. If FALSE, calculates for the entire dataset.
#' @param plotgrp : A character vector; variables from 'plot' tables for grouping. Use \code{c()} to combine multiple variables.
#' @param treegrp : A character vector; variables from 'tree' tables for grouping. Use \code{c()} to combine multiple variables.
#' @param strat : A character vector; the variable used for post-stratification. In the National Forest Inventory of Korea, it is typically used by forest type.
#' @param stockedland : A logical flag (default TRUE); if TRUE, includes only stocked land. If FALSE, includes all land types.
#' 
#' @return A `data.frame` that includes CWD biomass for plot or study areas.
#' The structure depends on the input parameters:
#' - If `byplot = TRUE`, each row represents a plot.
#' - If `byplot = FALSE`, each row represents the entire dataset or a group specified by `plotgrp` and `treegrp`
#' 
#' @note
#' Biomass calculation involves dividing the data into groups based on `plotgrp` and then applying post-stratification to each group. 
#' As a result, if the data for each group is not sufficiently large, the relative standard error (RSE) may be high. 
#' It is important to check the RSE and other statistical measures in the biomass results.
#' 
#' @examples
#' 
#' data("nfi_donghae")
#' 
#' # Basic usage
#' cwd <- cwd_biomass_nfi(nfi_donghae)
#' 
#' # Calculate CWD biomass grouped by administrative district and decay class
#' cwd_grp <- cwd_biomass_nfi(nfi_donghae, plotgrp = "SGG", treegrp = "DECAY")
#' 
#' # Calculate CWD biomass for each plot
#' plot_biomass <- cwd_biomass_nfi(nfi_donghae, byplot = TRUE)
#' 
#' @references 
#' Son, Y., Kim, R., Lee, K., Pyo, J., Kim, S., Hwang, J., Lee, S., & Park, H. (2014). Carbon emission factors and biomass allometric equations by species in Korea. Korea Forest Research Institute.
#' Yim, J., Moon, G., Lee, M., Kang, J., Won, M., Ahn, E., & Jeon, J. (2021). 2020 Forest inventory of Korea. Korea Forest Research Institute.
#' 
#' @export 


cwd_biomass_nfi <- function(data, byplot= FALSE, plotgrp=NULL, treegrp=NULL, strat="FORTYP_SUB", stockedland=TRUE){
  
  
  ## error message-------------------------------------------------------------- 
  required_names <- c("plot", "cwd")
  
  if (!all(required_names %in% names(data))) {
    missing_dfs <- required_names[!required_names %in% names(data)]
    stop("Missing required data frames in the list: ", paste(missing_dfs, collapse = ", "), call. = FALSE)
  }
  
  if (!is.null(plotgrp)){
    if(any(plotgrp %in% strat)){
      warning("param 'plotgrp' is the same as param 'strat'")
      strat <- "strat"
      data$plot$strat <- "same"
    }
    if(!is.character(plotgrp)) {
      stop("param 'plotgrp' must be 'character'")
    }
    if(any(!plotgrp %in% names(data$plot))){
      stop(paste0("param 'plotgrp': ", plotgrp," is not a column name in the 'plot' data frame."))
    }
  }
  
  
  if (!is.null(treegrp)){
    if(!is.character(treegrp)) {
      stop("param 'treegrp' must be 'character'")
    }
    if(any(!treegrp %in% names(data$cwd))){
      stop(paste0("param 'treegrp': ",treegrp," is not a column name in the 'cwd' data frame."))
    }
  }
  
  if (!is.null(strat)){
    if(!is.character(strat)) {
      stop("param 'strat' must be 'character'")
    }
    if(byplot){
      warning("param 'byplot' has priority over param 'strat'")
    }
    if(!strat %in% names(data$plot)){
      stop(paste0("param 'strat': ", strat," is not a column name in the 'plot' data frame."))
    }
    
  }
  
  
  ## Preprocessing--------------------------------------------------------------
  if (stockedland){ 
    data <- filter_nfi(data, c("plot$LAND_USECD == '1'"))
  }

  df <- left_join(data$cwd[, c('CLST_PLOT', 'SUB_PLOT',"CYCLE", 'SP', 'SPCD','DECAYCD', 
                                'CONDEC_CLASS_CD', 'VOL', treegrp)], 
                  data$plot[,c('CLST_PLOT', 'SUB_PLOT', "CYCLE", 'INVYR', 'LAND_USECD',
                               'NONFR_INCL_AREA_SUBP', strat, plotgrp)],
                  by = c("CLST_PLOT", "SUB_PLOT", "CYCLE"))
  
  if (!is.numeric(df$VOL)){
    df$VOL <- as.numeric(df$VOL)
  } 

  plot_id <- 'SUB_PLOT'
  df$tree_area <- 0.04 - ((df$NONFR_INCL_AREA_SUBP*10)/10000)
  df <- cwd_bm_df(df)
  
  plot_id  <- rlang::sym(plot_id)
  plotgrp  <- rlang::syms(plotgrp)
  treegrp  <- rlang::syms(treegrp)
  strat<- rlang::sym(strat)
  
  
  if(byplot){
    strat <- NULL
  }
  
  
  # 1. CWD Biomass calculation by subplot
  bm_temp <- df %>% 
    group_by(CYCLE, !!plot_id, INVYR, !!strat, !!!plotgrp, !!!treegrp, tree_area) %>% 
    summarise(cwd_volume_m3 = sum(VOL, na.rm=TRUE),
              cwd_carbon_stock_tC = sum(carbon_stock, na.rm=TRUE),
              cwd_co2_stock_tCO2 = sum(co2_stock, na.rm=TRUE),.groups = 'drop')
  

  condition_ha <- c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha", "cwd_co2_stock_tCO2_ha")
  bm_temp[condition_ha] <-  NA
  bm_temp <- as.data.frame(bm_temp)
  
  condition_ha <- (names(bm_temp) %in% c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha", "cwd_co2_stock_tCO2_ha"))
  condition <- (names(bm_temp) %in% c("cwd_volume_m3","cwd_carbon_stock_tC", "cwd_co2_stock_tCO2"))
  
  bm_temp[condition_ha] <- 
    lapply(bm_temp[condition], function(x) (x/bm_temp$tree_area))
  
  
  condition <- (names(bm_temp) %in% c("cwd_volume_m3","cwd_carbon_stock_tC", "cwd_co2_stock_tCO2"))
  
  bm_temp[condition] <- NULL
  bm_temp$tree_area <- NULL
  
  
  
  
  if(!byplot){ # 2.1 CWD Biomass calculation using post-stratified double sampling and weighted moving average methods
    
    # 2.1.1 Double sampling for post-strat(forest stand)
    weight_plotgrp <- data$plot %>% 
      group_by(CYCLE, !!!plotgrp) %>% 
      summarise(plot_num_all = n(),.groups = 'drop')
    
    
    weight_year <- data$plot %>% 
      group_by(CYCLE, INVYR, !!!plotgrp) %>% 
      summarise(plot_num_year = n(),.groups = 'drop')
    
    
    weight_stand <- data$plot %>% 
      group_by(CYCLE, INVYR, !!strat, !!!plotgrp) %>% 
      summarise(plot_num_stand = n(),.groups = 'drop')
    
    
    
    weight_DSS <- full_join(weight_stand, weight_year, by =c("CYCLE", "INVYR", as.character(unlist(lapply(plotgrp, quo_name)))))
    weight_DSS$weight_DSS <- weight_DSS$plot_num_stand/weight_DSS$plot_num_year
    
    
    # 2.1.2 Aggregating calculated biomass per plot by forest type.
    bm_temp_DSS <- bm_temp %>% 
      group_by(CYCLE, INVYR, !!strat, !!!plotgrp, !!!treegrp) %>% 
      summarise(var_cwd_volume_m3_ha =  var(cwd_volume_m3_ha, na.rm=TRUE),
                cwd_volume_m3_ha = sum(cwd_volume_m3_ha, na.rm=TRUE),
                var_cwd_carbon_stock_tC_ha =  var(cwd_carbon_stock_tC_ha, na.rm=TRUE),
                cwd_carbon_stock_tC_ha = sum(cwd_carbon_stock_tC_ha, na.rm=TRUE), 
                var_cwd_co2_stock_tCO2_ha =  var(cwd_co2_stock_tCO2_ha, na.rm=TRUE),
                cwd_co2_stock_tCO2_ha = sum(cwd_co2_stock_tCO2_ha, na.rm=TRUE),.groups = 'drop')
    
    
    bm_temp_DSS <- full_join(bm_temp_DSS, weight_DSS, by =c("CYCLE", "INVYR", quo_name(strat), as.character(unlist(lapply(plotgrp, quo_name)))))
    
    
    condition_DSS <- c("w_cwd_volume_m3_ha","w_cwd_carbon_stock_tC_ha", "w_cwd_co2_stock_tCO2_ha")
    bm_temp_DSS[condition_DSS] <-  NA
    bm_temp_DSS <- as.data.frame(bm_temp_DSS)
    
    condition <- (names(bm_temp_DSS) %in% c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha", "cwd_co2_stock_tCO2_ha"))
    
    bm_temp_DSS[condition] <- 
      lapply(bm_temp_DSS[condition], function(x) ((x/bm_temp_DSS$plot_num_stand)))
    
    condition_DSS <- (names(bm_temp_DSS) %in% c("w_cwd_volume_m3_ha","w_cwd_carbon_stock_tC_ha", "w_cwd_co2_stock_tCO2_ha"))
    bm_temp_DSS[condition_DSS] <- 
      lapply(bm_temp_DSS[condition], function(x) (x*bm_temp_DSS$weight_DSS))
    
    
    condition_var <- (names(bm_temp_DSS) %in% c("var_cwd_volume_m3_ha","var_cwd_carbon_stock_tC_ha", "var_cwd_co2_stock_tCO2_ha"))
    bm_temp_DSS[condition_var] <- 
      lapply(bm_temp_DSS[condition_var], function(x) ((bm_temp_DSS$weight_DSS)^2*(x/bm_temp_DSS$plot_num_stand)))
    
    
    bm_temp_WMA <- bm_temp_DSS %>% 
      group_by(CYCLE, INVYR, !!!plotgrp, !!!treegrp) %>% 
      summarise(w_cwd_volume_m3_ha = sum(w_cwd_volume_m3_ha, na.rm=TRUE),
                w_cwd_carbon_stock_tC_ha = sum(w_cwd_carbon_stock_tC_ha, na.rm=TRUE),
                w_cwd_co2_stock_tCO2_ha = sum(w_cwd_co2_stock_tCO2_ha, na.rm=TRUE),.groups = 'drop')
    
    
    # 2.1.3 Aggregating calculated biomass per forest type by study area.
    bm_temp_DSS[condition_DSS] <-  NULL
    bm_temp_DSS <- left_join(bm_temp_DSS, bm_temp_WMA, by =c("CYCLE", "INVYR", as.character(unlist(lapply(plotgrp, quo_name))),
                                                             as.character(unlist(lapply(treegrp, quo_name)))))

    bm_temp_DSS$var_cwd_volume_m3_ha <- bm_temp_DSS$var_cwd_volume_m3_ha + (bm_temp_DSS$weight_DSS*(bm_temp_DSS$cwd_volume_m3_ha-bm_temp_DSS$w_cwd_volume_m3_ha)^2/bm_temp_DSS$plot_num_year)
    bm_temp_DSS$var_cwd_carbon_stock_tC_ha <- bm_temp_DSS$var_cwd_carbon_stock_tC_ha + (bm_temp_DSS$weight_DSS*(bm_temp_DSS$cwd_carbon_stock_tC_ha-bm_temp_DSS$w_cwd_carbon_stock_tC_ha)^2/bm_temp_DSS$plot_num_year)
    bm_temp_DSS$var_cwd_co2_stock_tCO2_ha <- bm_temp_DSS$var_cwd_co2_stock_tCO2_ha + (bm_temp_DSS$weight_DSS*(bm_temp_DSS$cwd_co2_stock_tCO2_ha-bm_temp_DSS$w_cwd_co2_stock_tCO2_ha)^2/bm_temp_DSS$plot_num_year)
    
    condition <- (names(bm_temp_DSS) %in% c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha", "cwd_co2_stock_tCO2_ha"))
    condition_DSS <- (names(bm_temp_DSS) %in% c("w_cwd_volume_m3_ha","w_cwd_carbon_stock_tC_ha", "w_cwd_co2_stock_tCO2_ha"))
    
    bm_temp_DSS[condition_DSS] <- 
      lapply(bm_temp_DSS[condition], function(x) (x*bm_temp_DSS$weight_DSS))
    
    
    bm_temp_WMA <- bm_temp_DSS %>% 
      group_by(CYCLE, INVYR, !!!plotgrp, !!!treegrp) %>% 
      summarise(cwd_volume_m3_ha = sum(w_cwd_volume_m3_ha, na.rm=TRUE),
                var_cwd_volume_m3_ha = sum(var_cwd_volume_m3_ha, na.rm=TRUE),
                cwd_carbon_stock_tC_ha = sum(w_cwd_carbon_stock_tC_ha, na.rm=TRUE),
                var_cwd_carbon_stock_tC_ha = sum(var_cwd_carbon_stock_tC_ha, na.rm=TRUE),
                cwd_co2_stock_tCO2_ha = sum(w_cwd_co2_stock_tCO2_ha, na.rm=TRUE),
                var_cwd_co2_stock_tCO2_ha = sum(var_cwd_co2_stock_tCO2_ha, na.rm=TRUE), .groups = 'drop')
    
    
    # 2.1.4 Weighted Moving Average(to combine annual inventory field data)
    weight_WMA <- full_join(weight_year, weight_plotgrp, by =c("CYCLE", as.character(unlist(lapply(plotgrp, quo_name))))) 
    weight_WMA$weight_WMA <- weight_WMA$plot_num_year/weight_WMA$plot_num_all
    
    
    bm_temp_WMA <- full_join(bm_temp_WMA, weight_WMA, by =c("CYCLE","INVYR", as.character(unlist(lapply(plotgrp, quo_name)))))
    
    condition <- (names(bm_temp_WMA) %in% c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha"))
    
    
    bm_temp_WMA[condition] <- 
      lapply(bm_temp_WMA[condition], function(x) (x*bm_temp_WMA$weight_WMA))
    
    
    bm <- bm_temp_WMA %>% 
      group_by(CYCLE, !!!plotgrp, !!!treegrp) %>% 
      summarise(cwd_volume_m3_ha = sum(cwd_volume_m3_ha, na.rm=TRUE),
                var_cwd_volume_m3_ha = sum(weight_WMA^2*var_cwd_volume_m3_ha, na.rm=TRUE),
                se_cwd_volume_m3_ha = sqrt(var_cwd_volume_m3_ha),
                rse_cwd_volume_m3_ha = se_cwd_volume_m3_ha/cwd_volume_m3_ha*100,
                cwd_carbon_stock_tC_ha = sum(cwd_carbon_stock_tC_ha, na.rm=TRUE),
                var_cwd_carbon_stock_tC_ha = sum(weight_WMA^2*var_cwd_carbon_stock_tC_ha, na.rm=TRUE),
                se_cwd_carbon_stock_tC_ha = sqrt(var_cwd_carbon_stock_tC_ha),
                rse_cwd_carbon_stock_tC_ha = se_cwd_carbon_stock_tC_ha/cwd_carbon_stock_tC_ha*100,
                cwd_co2_stock_tCO2_ha = sum(cwd_co2_stock_tCO2_ha, na.rm=TRUE),
                var_cwd_co2_stock_tCO2_ha = sum(weight_WMA^2*var_cwd_co2_stock_tCO2_ha, na.rm=TRUE),
                se_cwd_co2_stock_tCO2_ha = sqrt(var_cwd_co2_stock_tCO2_ha),
                rse_cwd_co2_stock_tCO2_ha = se_cwd_co2_stock_tCO2_ha/cwd_co2_stock_tCO2_ha*100, .groups = 'drop')
    
    
    
    
    
  }else{ # 2.2 CWD Biomass calculation by plot
    
    bm <- bm_temp
    
  }
  
  
  
  
  
  
  
  return(bm)
  
}



#' cwd_biomass_tsvis()
#' 
#' @description
#' cwd_biomass_tsvis() function estimates volume, carbon storage and carbon dioxide storage of Coarse Woody Debris (CWD).
#' It is an internal function used within the tsvis_nfi() function.
#' cwd_biomass_nfi() calculates biomass by cycle, while this function calculates biomass by 5 year.
#'
#' @param data : A `list` generated by \code{\link{read_nfi}} that contains 'plot' and 'cwd' data frames.
#' @param plotgrp : A character vector; variables from 'plot' tables for grouping. Use \code{c()} to combine multiple variables.
#' @param treegrp : A character vector; variables from 'tree' tables for grouping. Use \code{c()} to combine multiple variables.
#' @param strat : A character vector; the variable used for post-stratification. In the National Forest Inventory of Korea, it is typically used by forest type.
#' @param stockedland : A logical flag (default TRUE); if TRUE, includes only stocked land. If FALSE, includes all land types.
#' 
#' @noRd



cwd_biomass_tsvis <- function(data, plotgrp=NULL, treegrp=NULL, strat="FORTYP_SUB", stockedland=TRUE){
  
  
  ## error message--------------------------------------------------------------  
  required_names <- c("plot", "cwd")
  
  if (!all(required_names %in% names(data))) {
    missing_dfs <- required_names[!required_names %in% names(data)]
    stop("Missing required data frames in the list: ", paste(missing_dfs, collapse = ", "), call. = FALSE)
  }
  
  if (!is.null(plotgrp)){
    if(any(plotgrp %in% strat)){
      warning("param 'plotgrp' is the same as param 'strat'")
      strat <- "strat"
      data$plot$strat <- "same"
    }
    if(!is.character(plotgrp)) {
      stop("param 'plotgrp' must be 'character'")
    }
    if(any(!plotgrp %in% names(data$plot))){
      stop(paste0("param 'plotgrp': ", plotgrp," is not a column name in the 'plot' data frame."))
    }
  }
  
  if (!is.null(treegrp)){
    if(!is.character(treegrp)) {
      stop("param 'treegrp' must be 'character'")
    }
    if(any(!treegrp %in% names(data$cwd))){
      stop(paste0("param 'treegrp': ",treegrp," is not a column name in the 'cwd' data frame."))
    }
  }
  
  if (!is.null(strat)){
    if(!is.character(strat)) {
      stop("param 'strat' must be 'character'")
    }
    if(!strat %in% names(data$plot)){
      stop(paste0("param 'strat': ", strat," is not a column name in the 'plot' data frame."))
    }
  }
  
  
  ## Preprocessing-------------------------------------------------------------- 
  if (stockedland){ 
    data <- filter_nfi(data, c("plot$LAND_USECD == '1'"))
  }
  
  df <- left_join(data$cwd[, c('CLST_PLOT', 'SUB_PLOT',"CYCLE", 'SP', 'SPCD', 'DECAYCD', 
                               'CONDEC_CLASS_CD', 'VOL', treegrp)], 
                  data$plot[,c('CLST_PLOT', 'SUB_PLOT', "CYCLE", 'INVYR', 'LAND_USECD',
                               'NONFR_INCL_AREA_SUBP', strat, plotgrp)],
                  by = c("CLST_PLOT", "SUB_PLOT", "CYCLE"))
  
  

  if (!is.numeric(df$VOL)){
    df$VOL <- as.numeric(df$VOL)
  } 

  
  plot_id <- 'SUB_PLOT'
  df$tree_area <- 0.04 - ((df$NONFR_INCL_AREA_SUBP*10)/10000)
  df <- cwd_bm_df(df)
  
  plot_id  <- rlang::sym(plot_id)
  plotgrp  <- rlang::syms(plotgrp)
  treegrp  <- rlang::syms(treegrp)
  strat<- rlang::sym(strat)
  
  
  # 1. CWD Biomass calculation by subplot
  bm_temp <- df %>% 
    group_by(CYCLE, !!plot_id, INVYR, !!strat, !!!plotgrp, !!!treegrp, tree_area) %>% 
    summarise(cwd_volume_m3 = sum(VOL, na.rm=TRUE),
              cwd_carbon_stock_tC = sum(carbon_stock, na.rm=TRUE),
              cwd_co2_stock_tCO2 = sum(co2_stock, na.rm=TRUE), .groups = 'drop')
  
  condition_ha <- c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha", "cwd_co2_stock_tCO2_ha")
  bm_temp[condition_ha] <-  NA
  bm_temp <- as.data.frame(bm_temp)
  
  condition_ha <- (names(bm_temp) %in% c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha", "cwd_co2_stock_tCO2_ha"))
  condition <- (names(bm_temp) %in% c("cwd_volume_m3","cwd_carbon_stock_tC", "cwd_co2_stock_tCO2"))
  
  bm_temp[condition_ha] <- 
    lapply(bm_temp[condition], function(x) (x/bm_temp$tree_area))
  
  condition <- (names(bm_temp) %in% c("cwd_volume_m3","cwd_carbon_stock_tC", "cwd_co2_stock_tCO2"))
  
  bm_temp[condition] <- NULL
  bm_temp$tree_area <- NULL

  
  # 2.1 CWD Biomass calculation using post-stratified double sampling and weighted moving average methods
  # 2.1.1 Double sampling for post-strat(forest stand)
  weight_plotgrp <- data$plot %>% 
    group_by(!!!plotgrp) %>% 
    summarise(plot_num_all = n(),.groups = 'drop')
  
  
  weight_year <- data$plot %>% 
    group_by(CYCLE, INVYR, !!!plotgrp) %>% 
    summarise(plot_num_year = n(),.groups = 'drop')
  
  
  weight_stand <- data$plot %>% 
    group_by(CYCLE, INVYR, !!strat, !!!plotgrp) %>% 
    summarise(plot_num_stand = n(),.groups = 'drop')
  
  weight_DSS <- full_join(weight_stand, weight_year, by =c("CYCLE", "INVYR", as.character(unlist(lapply(plotgrp, quo_name)))))
  weight_DSS$weight_DSS <- weight_DSS$plot_num_stand/weight_DSS$plot_num_year
  
  
  # 2.1.2 Aggregating calculated biomass per plot by forest type.
  bm_temp_DSS <- bm_temp %>% 
    group_by(CYCLE, INVYR, !!strat, !!!plotgrp, !!!treegrp) %>% 
    summarise(var_cwd_volume_m3_ha =  var(cwd_volume_m3_ha, na.rm=TRUE),
              cwd_volume_m3_ha = sum(cwd_volume_m3_ha, na.rm=TRUE),
              var_cwd_carbon_stock_tC_ha =  var(cwd_carbon_stock_tC_ha, na.rm=TRUE),
              cwd_carbon_stock_tC_ha = sum(cwd_carbon_stock_tC_ha, na.rm=TRUE), 
              var_cwd_co2_stock_tCO2_ha =  var(cwd_co2_stock_tCO2_ha, na.rm=TRUE),
              cwd_co2_stock_tCO2_ha = sum(cwd_co2_stock_tCO2_ha, na.rm=TRUE),.groups = 'drop')
  
  
  bm_temp_DSS <- full_join(bm_temp_DSS, weight_DSS, by =c("CYCLE", "INVYR", quo_name(strat), as.character(unlist(lapply(plotgrp, quo_name)))))
  
  
  condition_DSS <- c("w_cwd_volume_m3_ha","w_cwd_carbon_stock_tC_ha", "w_cwd_co2_stock_tCO2_ha")
  bm_temp_DSS[condition_DSS] <-  NA
  bm_temp_DSS <- as.data.frame(bm_temp_DSS)
  
  condition <- (names(bm_temp_DSS) %in% c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha", "cwd_co2_stock_tCO2_ha"))
  
  bm_temp_DSS[condition] <- 
    lapply(bm_temp_DSS[condition], function(x) ((x/bm_temp_DSS$plot_num_stand)))
  
  condition_DSS <- (names(bm_temp_DSS) %in% c("w_cwd_volume_m3_ha","w_cwd_carbon_stock_tC_ha", "w_cwd_co2_stock_tCO2_ha"))
  bm_temp_DSS[condition_DSS] <- 
    lapply(bm_temp_DSS[condition], function(x) (x*bm_temp_DSS$weight_DSS))
  
  
  condition_var <- (names(bm_temp_DSS) %in% c("var_cwd_volume_m3_ha","var_cwd_carbon_stock_tC_ha", "var_cwd_co2_stock_tCO2_ha"))
  bm_temp_DSS[condition_var] <- 
    lapply(bm_temp_DSS[condition_var], function(x) ((bm_temp_DSS$weight_DSS)^2*(x/bm_temp_DSS$plot_num_stand)))
  
  
  bm_temp_WMA <- bm_temp_DSS %>% 
    group_by(CYCLE, INVYR, !!!plotgrp, !!!treegrp) %>% 
    summarise(w_cwd_volume_m3_ha = sum(w_cwd_volume_m3_ha, na.rm=TRUE),
              w_cwd_carbon_stock_tC_ha = sum(w_cwd_carbon_stock_tC_ha, na.rm=TRUE),
              w_cwd_co2_stock_tCO2_ha = sum(w_cwd_co2_stock_tCO2_ha, na.rm=TRUE), .groups = 'drop')
  
  # 2.1.3 Aggregating calculated biomass per forest type by study area.
  bm_temp_DSS[condition_DSS] <-  NULL
  bm_temp_DSS <- left_join(bm_temp_DSS, bm_temp_WMA, by =c("CYCLE", "INVYR", as.character(unlist(lapply(plotgrp, quo_name))),
                                                           as.character(unlist(lapply(treegrp, quo_name)))))
  
  
  bm_temp_DSS$var_cwd_volume_m3_ha <- bm_temp_DSS$var_cwd_volume_m3_ha + (bm_temp_DSS$weight_DSS*(bm_temp_DSS$cwd_volume_m3_ha-bm_temp_DSS$w_cwd_volume_m3_ha)^2/bm_temp_DSS$plot_num_year)
  bm_temp_DSS$var_cwd_carbon_stock_tC_ha <- bm_temp_DSS$var_cwd_carbon_stock_tC_ha + (bm_temp_DSS$weight_DSS*(bm_temp_DSS$cwd_carbon_stock_tC_ha-bm_temp_DSS$w_cwd_carbon_stock_tC_ha)^2/bm_temp_DSS$plot_num_year)
  bm_temp_DSS$var_cwd_co2_stock_tCO2_ha <- bm_temp_DSS$var_cwd_co2_stock_tCO2_ha + (bm_temp_DSS$weight_DSS*(bm_temp_DSS$cwd_co2_stock_tCO2_ha-bm_temp_DSS$w_cwd_co2_stock_tCO2_ha)^2/bm_temp_DSS$plot_num_year)
  
  condition <- (names(bm_temp_DSS) %in% c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha", "cwd_co2_stock_tCO2_ha"))
  condition_DSS <- (names(bm_temp_DSS) %in% c("w_cwd_volume_m3_ha","w_cwd_carbon_stock_tC_ha", "w_cwd_co2_stock_tCO2_ha"))
  
  bm_temp_DSS[condition_DSS] <- 
    lapply(bm_temp_DSS[condition], function(x) (x*bm_temp_DSS$weight_DSS))
  
  
  bm_temp_WMA <- bm_temp_DSS %>% 
    group_by(CYCLE, INVYR, !!!plotgrp, !!!treegrp) %>% 
    summarise(cwd_volume_m3_ha = sum(w_cwd_volume_m3_ha, na.rm=TRUE),
              var_cwd_volume_m3_ha = sum(var_cwd_volume_m3_ha, na.rm=TRUE),
              cwd_carbon_stock_tC_ha = sum(w_cwd_carbon_stock_tC_ha, na.rm=TRUE),
              var_cwd_carbon_stock_tC_ha = sum(var_cwd_carbon_stock_tC_ha, na.rm=TRUE),
              cwd_co2_stock_tCO2_ha = sum(w_cwd_co2_stock_tCO2_ha, na.rm=TRUE),
              var_cwd_co2_stock_tCO2_ha = sum(var_cwd_co2_stock_tCO2_ha, na.rm=TRUE), .groups = 'drop')
  

  # 2.1.4 Weighted Moving Average(to combine annual inventory
  weight_WMA <- full_join(weight_year, weight_plotgrp, by =c(as.character(unlist(lapply(plotgrp, quo_name)))))
  weight_WMA$weight_WMA <- weight_WMA$plot_num_year/weight_WMA$plot_num_all
  
  
  bm_temp_WMA <- full_join(bm_temp_WMA, weight_WMA, by =c("CYCLE","INVYR", as.character(unlist(lapply(plotgrp, quo_name)))))
  
  condition <- (names(bm_temp_WMA) %in% c("cwd_volume_m3_ha","cwd_carbon_stock_tC_ha", "cwd_co2_stock_tCO2_ha"))
  
  
  bm_temp_WMA[condition] <- 
    lapply(bm_temp_WMA[condition], function(x) (x*bm_temp_WMA$weight_WMA))
  
  
  bm <- bm_temp_WMA %>% 
    group_by(!!!plotgrp, !!!treegrp) %>% 
    summarise(cwd_volume_m3_ha = sum(cwd_volume_m3_ha, na.rm=TRUE),
              var_cwd_volume_m3_ha = sum(weight_WMA^2*var_cwd_volume_m3_ha, na.rm=TRUE),
              se_cwd_volume_m3_ha = sqrt(var_cwd_volume_m3_ha),
              rse_cwd_volume_m3_ha = se_cwd_volume_m3_ha/cwd_volume_m3_ha*100,
              cwd_carbon_stock_tC_ha = sum(cwd_carbon_stock_tC_ha, na.rm=TRUE),
              var_cwd_carbon_stock_tC_ha = sum(weight_WMA^2*var_cwd_carbon_stock_tC_ha, na.rm=TRUE),
              se_cwd_carbon_stock_tC_ha = sqrt(var_cwd_carbon_stock_tC_ha),
              rse_cwd_carbon_stock_tC_ha = se_cwd_carbon_stock_tC_ha/cwd_carbon_stock_tC_ha*100,
              cwd_co2_stock_tCO2_ha = sum(cwd_co2_stock_tCO2_ha, na.rm=TRUE),
              var_cwd_co2_stock_tCO2_ha = sum(weight_WMA^2*var_cwd_co2_stock_tCO2_ha, na.rm=TRUE),
              se_cwd_co2_stock_tCO2_ha = sqrt(var_cwd_co2_stock_tCO2_ha),
              rse_cwd_co2_stock_tCO2_ha = se_cwd_co2_stock_tCO2_ha/cwd_co2_stock_tCO2_ha*100,.groups = 'drop')
  

  return(bm)
  
}


