% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/incidence.R
\name{incidence}
\alias{incidence}
\alias{incidence.integer}
\alias{incidence.default}
\alias{incidence.numeric}
\alias{incidence.Date}
\alias{incidence.POSIXt}
\alias{print.incidence}
\title{Compute incidence of events from a vector of dates.}
\usage{
incidence(dates, interval = 1L, ...)

\method{incidence}{integer}(dates, interval = 1L, groups = NULL,
  na_as_group = TRUE, ...)

\method{incidence}{default}(dates, interval = 1L, groups = NULL,
  na_as_group = TRUE, ...)

\method{incidence}{numeric}(dates, interval = 1L, ...)

\method{incidence}{Date}(dates, interval = 1L, iso_week = TRUE, ...)

\method{incidence}{POSIXt}(dates, interval = 1L, iso_week = TRUE, ...)

\method{print}{incidence}(x, ...)
}
\arguments{
\item{dates}{A vector of dates, which can be provided as objects of the
class: integer, numeric, Date, POSIXct. Note that decimal numbers will be
floored with a warning.}

\item{interval}{An integer indicating the (fixed) size of the time interval
used for computing the incidence; defaults to 1 day.}

\item{...}{Additional arguments passed to other methods (none are used).}

\item{groups}{An optional factor defining groups of observations for which
incidence should be computed separately.}

\item{na_as_group}{A logical value indicating if missing group (NA) should be
treated as a separate group.}

\item{iso_week}{A logical value indicating if the returning \code{incidence}
should be ISO week-based when computing weekly incidence (interval =
7). defaults to be TRUE.}

\item{x}{An 'incidence' object.}
}
\value{
An list with the class \code{incidence}, which contains the
following items:

\itemize{

\item dates: The dates marking the left side of the bins used for counting
events. When ISO week-based weekly incidence is computed, the dates are the
first days of corresponding isoweeks.

\item counts: A matrix of incidence counts, which one column per group (and
a single column if no groups were used).

\item timespan: The length of the period for which incidence is computed, in
days.

\item interval: The bin size, in number of days; e.g. 7 indicates weekly
incidence.

\item n: The total number of cases.

\item isoweeks: ISO 8601 week format yyyy-Www, which is returned only when
ISO week-based weekly incidence is computed.

}
}
\description{
This function computes incidence based on dates of events provided in
various formats. A fixed interval, provided as numbers of days, is used to
define time intervals. Counts within an interval always include the first
date, after which they are labelled, and exclude the second. For instance,
intervals labelled as 0, 3, 6, ... mean that the first bin includes days 0, 1
and 2, the second interval includes 3, 4 and 5 etc.
}
\details{
For details about the \code{incidence class}, see the dedicated
vignette:\cr \code{vignette("incidence_class", package = "incidence")}
}
\examples{
## toy example
incidence(c(1, 5, 8, 3, 7, 2, 4, 6, 9, 2))
incidence(c(1, 5, 8, 3, 7, 2, 4, 6, 9, 2), 2)

## example using simulated dataset
if(require(outbreaks)) {
  onset <- ebola_sim$linelist$date_of_onset

  ## daily incidence
  inc <- incidence(onset)
  inc
  plot(inc)

  ## weekly incidence
  inc.week <- incidence(onset, interval = 7, iso_week = FALSE)
  inc.week
  plot(inc.week)
  plot(inc.week, border = "white") # with visible border
  inc.isoweek <- incidence(onset, interval = 7, iso_week = TRUE)
  inc.isoweek
  ## use group information
  sex <- ebola_sim$linelist$gender
  inc.week.gender <- incidence(onset, interval = 7, groups = sex, iso_week = FALSE)
  inc.week.gender
  head(inc.week.gender$counts)
  plot(inc.week.gender)
  inc.isoweek.gender <- incidence(onset, interval = 7, groups = sex, iso_week = TRUE)
  inc.isoweek.gender
}


}
\seealso{
The main other functions of the package include:
\itemize{

 \item \code{\link{fit}}: Fit log-linear model to computed incidence.

 \item \code{\link{fit_optim_split}}: Find the optimal peak of the epidemic
and fits log-linear models on either side of the peak.

 \item \code{\link[incidence]{subset}}: Handling of \code{incidence}
objects.

 \item \code{\link{pool}}: Sum incidence over groups.

 \item \code{\link{as.data.frame}}: Convert an \code{incidence} object to a
\code{data.frame}.}

The following vignettes are also available:
\itemize{

\item \code{overview}: Provides an overview of the package's features.

\item \code{customize_plot}: Provides some tips on finer plot customization.

\item \code{incidence_class}: Details the content of the \code{incidence}
class.

}
}
\author{
Thibaut Jombart, Rich Fitzjohn
}
