% Generated by roxygen2 (4.0.1): do not edit by hand
\name{icpower}
\alias{icpower}
\title{Study design in the presence of error-prone diagnostic tests and
self-reported outcomes}
\usage{
icpower(HR, sensitivity, specificity, survivals, N = NULL, power = NULL,
  rho = 0.5, alpha = 0.05, pmiss = 0, design = "MCAR", negpred = 1)
}
\arguments{
\item{HR}{hazard ratio under the alternative hypothesis.}

\item{sensitivity}{the sensitivity of test.}

\item{specificity}{the specificity of test}

\item{survivals}{a vector of survival function at each test time for
baseline(reference) group. Its length determines the number of tests.}

\item{N}{a vector of sample sizes to calculate corresponding powers. If one
needs to calculate sample size, then set to NULL.}

\item{power}{a vector of powers to calculate corresponding sample sizes. If
one needs to calculate power, then set to NULL.}

\item{rho}{proportion of subjects in baseline(reference) group.}

\item{alpha}{type I error.}

\item{pmiss}{a value or a vector (must have same length as survivals) of the
probabilities of each test being randomly missing at each test time. If
pmiss is a single value, then each test is assumed to have an identical
probability of missingness.}

\item{design}{missing mechanism: "MCAR" or "NTFP".}

\item{negpred}{baseline negative predictive value, i.e. the probability of
  being truely disease free for those who were tested (reported) as disease
  free at baseline. If baseline screening test is perfect, then negpred = 1.}
}
\value{
\itemize{ \item result: a data frame with calculated sample size and
  power \item I1 and I2: calculated unit Fisher information matrices for each
  group, which can be used to calculate more values of sample size and power
  for the same design without the need to enumerate again }
}
\description{
This function calculates the power and sample in the presence of error-prone
diagnostic tests and self-reported outcomes.  Two missing mechanisms can be
assumed, namely MCAR and NTFP. The MCAR setting assumes that each test is
subject to a constant, independent probability of missingness. The NTFP
mechanism includes two types of missingness - (1) incorporates a constant,
independent, probability of missing for each test prior to the first positive
test result; and (2) all test results after first positive are missing.
}
\details{
To calculate sample sizes for a vector of powers, set N = NULL. To
  calculate powers for a vector of sample sizes, set power = NULL. One and
  only one of power and N should be specified, and the other set to NULL.
  This function uses an enumeration algorithm to calculate the expected
  Fisher information matrix. The expected Fisher information matrix is used
  to obtain the variance of the coefficient corresponding to the treatment
  group indicator.
}
\note{
When diagnostic test is perfect, i.e. sensitivity=1 and
  specificity=1, use \code{\link{icpowerpf}} instead to obtain significantly
  improved computational efficiency.
}
\examples{
## First specificy survivals. Assume test times are 1:8, with survival function
## at the end time 0.9
surv <- exp(log(0.9)*(1:8)/8)

## Obtain power vs. N
pow1 <- icpower(HR = 2, sensitivity = 0.55, specificity = 0.99, survivals = surv,
   N = seq(500, 1500, 50), power = NULL, rho = 0.5, alpha = 0.05,
   pmiss = 0, design = "MCAR", negpred = 1)

plot(pow1$result$N, pow1$result$power, type="l", xlab="N", ylab="power")

## Calculate sample size, assuming desired power is 0.9
pow2 <- icpower(HR = 2, sensitivity = 0.55, specificity = 0.99, survivals = surv,
   N = NULL, power = 0.9, rho = 0.5, alpha = 0.05, pmiss = 0, design = "MCAR",
   negpred = 1)

## When missing test is present with MCAR
pow3 <- icpower(HR = 2, sensitivity = 0.55, specificity = 0.99, survivals = surv,
   N = NULL, power = 0.9, rho = 0.5, alpha = 0.05, pmiss = 0.4, design = "MCAR",
   negpred = 1)

## When missing test is present with NTFP
pow4 <- icpower(HR = 2, sensitivity = 0.55, specificity = 0.99, survivals = surv,
   N = NULL, power = 0.9, rho = 0.5, alpha = 0.05, pmiss = 0.4, design = "NTFP",
   negpred = 1)

## When baseline misclassification is present
pow5 <- icpower(HR = 2, sensitivity = 0.55, specificity = 0.99, survivals = surv,
   N = NULL, power = 0.9, rho = 0.5, alpha = 0.05, pmiss = 0, design = "MCAR",
   negpred = 0.98)

## When test is  perfect and no missing test
pow6 <- icpower(HR = 2, sensitivity = 1, specificity = 1, survivals = surv,
   N = NULL, power = 0.9, rho = 0.5, alpha = 0.05, pmiss = 0, design = "MCAR",
   negpred = 1)

## Different missing probabilities at each test time
pow7 <- icpower(HR = 2, sensitivity = 0.55, specificity = 0.99, survivals = surv,
   N = NULL, power = 0.9, rho = 0.5, alpha = 0.05, pmiss = seq(0.1, 0.8, 0.1),
   design = "MCAR")
}
\seealso{
\code{\link{icpowerpf}}
}

